/**
 * DBG Payments - WooCommerce Blocks Checkout Integration
 * Enables Linden Dollar (L$) payments from Second Life users
 */
const { registerPaymentMethod } = window.wc.wcBlocksRegistry;
const { createElement, useState, useEffect } = window.wp.element;
const { decodeEntities } = window.wp.htmlEntities;
const { getSetting } = window.wc.wcSettings;

// Get settings from PHP
const settings = getSetting('secondlife_data', {});
const defaultTitle = 'Pay with Linden Dollars (L$)';
const title = decodeEntities(settings.title || defaultTitle);
const description = decodeEntities(settings.description || '');

// Currency conversion settings
const lindenRate = settings.lindenRate || settings.exchangeRate || 250; // L$ per $1 USD
const currencyToUsdFactor = settings.currencyToUsdFactor || 1.0; // Store currency to USD conversion
const storeCurrency = settings.currency || 'USD';

/**
 * Content component - shown when payment method is selected
 */
const Content = (props) => {
    const { eventRegistration, emitResponse } = props;
    const { onPaymentSetup } = eventRegistration;
    
    const [firstName, setFirstName] = useState('');
    const [lastName, setLastName] = useState('');
    const [error, setError] = useState('');
    
    // Calculate L$ amount from cart total
    // First convert from store currency to USD, then to L$
    const cartTotal = props.billing?.cartTotal?.value || 0;
    const cartTotalFloat = cartTotal / 100; // WooCommerce stores in cents
    const usdAmount = cartTotalFloat * currencyToUsdFactor; // Convert to USD
    const lindenAmount = Math.ceil(usdAmount * lindenRate); // Convert to L$
    
    useEffect(() => {
        const unsubscribe = onPaymentSetup(() => {
            // Validate SL name
            if (!firstName.trim()) {
                setError('Please enter your Second Life first name.');
                return {
                    type: emitResponse.responseTypes.ERROR,
                    message: 'Please enter your Second Life first name.',
                };
            }
            
            if (!lastName.trim()) {
                setError('Please enter your Second Life last name.');
                return {
                    type: emitResponse.responseTypes.ERROR,
                    message: 'Please enter your Second Life last name.',
                };
            }
            
            // Basic validation - SL names are alphanumeric
            const namePattern = /^[a-zA-Z0-9]+$/;
            if (!namePattern.test(firstName)) {
                setError('Invalid Second Life first name. Use only letters and numbers.');
                return {
                    type: emitResponse.responseTypes.ERROR,
                    message: 'Invalid Second Life first name.',
                };
            }
            
            if (!namePattern.test(lastName)) {
                setError('Invalid Second Life last name. Use only letters and numbers.');
                return {
                    type: emitResponse.responseTypes.ERROR,
                    message: 'Invalid Second Life last name.',
                };
            }
            
            setError('');
            
            // Return payment method data
            return {
                type: emitResponse.responseTypes.SUCCESS,
                meta: {
                    paymentMethodData: {
                        sl_first_name: firstName.trim(),
                        sl_last_name: lastName.trim(),
                    },
                },
            };
        });
        
        return () => unsubscribe();
    }, [firstName, lastName, onPaymentSetup, emitResponse]);
    
    return createElement('div', { className: 'sl-payment-fields-blocks' },
        // L$ Amount display
        createElement('div', { 
            className: 'sl-linden-amount',
            style: { 
                padding: '12px', 
                background: '#f0f0f1', 
                borderRadius: '4px',
                marginBottom: '16px',
                borderLeft: '4px solid #2271b1'
            }
        },
            createElement('strong', null, 'Amount in Linden Dollars: '),
            createElement('span', { style: { fontSize: '18px', fontWeight: 'bold' } }, 
                'L$' + lindenAmount.toLocaleString()
            )
        ),
        
        // Description
        description && createElement('p', { 
            className: 'sl-payment-description',
            style: { marginBottom: '16px' }
        }, description),
        
        // Error message
        error && createElement('div', {
            style: {
                padding: '10px',
                background: '#fcf0f1',
                border: '1px solid #d63638',
                borderRadius: '4px',
                marginBottom: '16px',
                color: '#d63638'
            }
        }, error),
        
        // SL Name - Single input for full avatar name
        createElement('div', { 
            className: 'sl-name-field',
            style: { marginBottom: '16px' }
        },
            createElement('label', { 
                htmlFor: 'sl-avatar-name',
                style: { display: 'block', marginBottom: '8px', fontWeight: '600' }
            }, 'Second Life Avatar Name *'),
            createElement('div', {
                style: { display: 'flex', gap: '8px', alignItems: 'center' }
            },
                createElement('input', {
                    type: 'text',
                    id: 'sl-first-name',
                    value: firstName,
                    onChange: (e) => setFirstName(e.target.value),
                    placeholder: 'First name',
                    required: true,
                    style: { 
                        flex: 1,
                        padding: '12px', 
                        border: '1px solid #8c8f94', 
                        borderRadius: '4px',
                        boxSizing: 'border-box',
                        fontSize: '16px'
                    }
                }),
                createElement('input', {
                    type: 'text',
                    id: 'sl-last-name',
                    value: lastName,
                    onChange: (e) => setLastName(e.target.value),
                    placeholder: 'Resident',
                    required: true,
                    style: { 
                        flex: 1,
                        padding: '12px', 
                        border: '1px solid #8c8f94', 
                        borderRadius: '4px',
                        boxSizing: 'border-box',
                        fontSize: '16px'
                    }
                })
            ),
            createElement('p', {
                style: { fontSize: '12px', color: '#646970', marginTop: '8px', marginBottom: 0 }
            }, 'Enter your avatar name as it appears in Second Life (e.g., "John Resident")')
        ),
        
        // Help text
        createElement('p', { 
            className: 'sl-payment-note',
            style: { fontSize: '13px', color: '#646970', margin: 0, fontStyle: 'italic' }
        }, 
            'After placing your order, you will be directed to pay at our in-world location.'
        )
    );
};

/**
 * Label component - payment method name in the list
 */
const Label = (props) => {
    const { PaymentMethodLabel } = props.components;
    return createElement(PaymentMethodLabel, { text: title });
};

/**
 * Register the payment method
 */
registerPaymentMethod({
    name: 'secondlife',
    label: createElement(Label, null),
    content: createElement(Content, null),
    edit: createElement(Content, null),
    canMakePayment: () => true,
    ariaLabel: title,
    supports: {
        features: settings.supports || ['products'],
    },
});
