/**
 * DBG Payments - Checkout JavaScript
 * Handles payment status polling and UI updates
 */

(function($) {
    'use strict';

    var SLPayments = {
        pollInterval: null,
        pollDelay: 3000, // 3 seconds
        maxPolls: 300, // 15 minutes at 3 second intervals
        pollCount: 0,
        transactionId: null,
        expiresAt: null,
        countdownInterval: null,

        /**
         * Initialize payment polling
         */
        init: function() {
            var $pending = $('.sl-payment-pending');
            
            if ($pending.length === 0) {
                return;
            }

            this.transactionId = $pending.data('transaction-id');
            
            if (!this.transactionId) {
                return;
            }

            // Get expiration time
            var $countdown = $('.sl-countdown');
            if ($countdown.length) {
                this.expiresAt = parseInt($countdown.data('expires'), 10);
            }

            // Start polling for status
            this.startPolling();

            // Start countdown timer
            this.startCountdown();

            console.log('[SL Payments] Waiting for payment:', this.transactionId);
        },

        /**
         * Start polling for payment status
         */
        startPolling: function() {
            var self = this;
            
            this.pollInterval = setInterval(function() {
                self.checkStatus();
            }, this.pollDelay);

            // Also check immediately
            this.checkStatus();
        },

        /**
         * Stop polling
         */
        stopPolling: function() {
            if (this.pollInterval) {
                clearInterval(this.pollInterval);
                this.pollInterval = null;
            }
        },

        /**
         * Check payment status
         */
        checkStatus: function() {
            var self = this;

            this.pollCount++;

            // Check if we've exceeded max polls
            if (this.pollCount > this.maxPolls) {
                this.handleTimeout();
                return;
            }

            $.ajax({
                url: slPayments.restUrl + 'status/' + this.transactionId,
                method: 'GET',
                headers: {
                    'X-WP-Nonce': slPayments.nonce
                },
                success: function(response) {
                    self.handleStatusResponse(response);
                },
                error: function(xhr, status, error) {
                    console.error('[SL Payments] Status check failed:', error);
                }
            });
        },

        /**
         * Handle status response
         */
        handleStatusResponse: function(response) {
            console.log('[SL Payments] Status:', response.status);

            switch (response.status) {
                case 'completed':
                    this.handleCompleted();
                    break;
                    
                case 'expired':
                    this.handleExpired();
                    break;
                    
                case 'failed':
                    this.handleFailed(response.error_message || '');
                    break;
                    
                case 'pending':
                    // Still waiting, continue polling
                    break;
            }
        },

        /**
         * Handle successful payment
         */
        handleCompleted: function() {
            this.stopPolling();
            this.stopCountdown();

            var $pending = $('.sl-payment-pending');
            var $status = $pending.find('.sl-payment-status');
            
            // Update UI
            $pending.addClass('sl-payment-completed');
            $status.find('h3').text(slPayments.i18n.completed);
            $pending.find('.sl-payment-instructions, .sl-payment-timeout').fadeOut();

            // Show success message
            $('<div class="sl-payment-success-msg">')
                .html('<p>✓ ' + slPayments.i18n.completed + '</p>')
                .insertAfter($status);

            // Reload page after short delay to show updated order status
            setTimeout(function() {
                window.location.reload();
            }, 2000);
        },

        /**
         * Handle expired payment
         */
        handleExpired: function() {
            this.stopPolling();
            this.stopCountdown();

            var $pending = $('.sl-payment-pending');
            var $status = $pending.find('.sl-payment-status');
            
            // Update UI
            $pending.addClass('sl-payment-expired');
            $status.find('.sl-spinner').hide();
            $status.find('h3').text(slPayments.i18n.timeout);

            // Show retry option
            $('<div class="sl-payment-retry">')
                .html('<p>' + slPayments.i18n.timeout + '</p><button class="button" onclick="window.location.reload()">Try Again</button>')
                .insertAfter($status);
        },

        /**
         * Handle failed payment
         */
        handleFailed: function(message) {
            this.stopPolling();
            this.stopCountdown();

            var $pending = $('.sl-payment-pending');
            var $status = $pending.find('.sl-payment-status');
            
            // Update UI
            $pending.addClass('sl-payment-failed');
            $status.find('.sl-spinner').hide();
            $status.find('h3').text(slPayments.i18n.error);

            if (message) {
                $('<p class="sl-error-message">').text(message).insertAfter($status.find('h3'));
            }
        },

        /**
         * Handle polling timeout
         */
        handleTimeout: function() {
            this.handleExpired();
        },

        /**
         * Start countdown timer
         */
        startCountdown: function() {
            var self = this;
            var $countdown = $('.sl-countdown');
            
            if (!this.expiresAt || $countdown.length === 0) {
                return;
            }

            this.countdownInterval = setInterval(function() {
                self.updateCountdown($countdown);
            }, 1000);
        },

        /**
         * Stop countdown
         */
        stopCountdown: function() {
            if (this.countdownInterval) {
                clearInterval(this.countdownInterval);
                this.countdownInterval = null;
            }
        },

        /**
         * Update countdown display
         */
        updateCountdown: function($countdown) {
            var now = Math.floor(Date.now() / 1000);
            var remaining = this.expiresAt - now;

            if (remaining <= 0) {
                $countdown.text('0:00');
                this.handleExpired();
                return;
            }

            var minutes = Math.floor(remaining / 60);
            var seconds = remaining % 60;
            
            $countdown.text(minutes + ':' + (seconds < 10 ? '0' : '') + seconds);

            // Add warning class when low on time
            if (remaining < 60) {
                $countdown.addClass('sl-countdown-warning');
            }
        }
    };

    // Initialize when document is ready
    $(document).ready(function() {
        SLPayments.init();
    });

    // Expose for debugging
    window.SLPayments = SLPayments;

})(jQuery);
