<?php
/**
 * Plugin activation handler
 * Creates database tables and sets up default options
 */

if (!defined('WPINC')) {
    die;
}

class SL_Activator {
    
    /**
     * Plugin activation
     */
    public static function activate() {
        self::create_tables();
        self::set_default_options();
        flush_rewrite_rules();
    }
    
    /**
     * Create database tables
     */
    private static function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Boxes table - stores registered payment boxes
        $table_boxes = $wpdb->prefix . 'sl_boxes';
        $sql_boxes = "CREATE TABLE $table_boxes (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            api_key varchar(64) NOT NULL,
            owner_key varchar(36) NOT NULL,
            owner_name varchar(128) DEFAULT '',
            object_name varchar(255) DEFAULT '',
            url text DEFAULT '',
            region varchar(255) DEFAULT '',
            position varchar(64) DEFAULT '',
            is_active tinyint(1) DEFAULT 1,
            last_seen datetime DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY api_key (api_key),
            KEY owner_key (owner_key),
            KEY is_active (is_active)
        ) $charset_collate;";
        
        // Transactions table - stores payment records
        $table_transactions = $wpdb->prefix . 'sl_transactions';
        $sql_transactions = "CREATE TABLE $table_transactions (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            transaction_id varchar(64) NOT NULL,
            box_id bigint(20) unsigned DEFAULT NULL,
            order_id bigint(20) unsigned DEFAULT NULL,
            buyer_sl_name varchar(128) NOT NULL,
            buyer_sl_key varchar(36) DEFAULT '',
            amount int(11) NOT NULL,
            fee int(11) DEFAULT 0,
            status varchar(32) DEFAULT 'pending',
            product_data text DEFAULT '',
            error_message text DEFAULT '',
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            completed_at datetime DEFAULT NULL,
            expires_at datetime DEFAULT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY transaction_id (transaction_id),
            KEY box_id (box_id),
            KEY order_id (order_id),
            KEY status (status),
            KEY buyer_sl_name (buyer_sl_name),
            KEY expires_at (expires_at)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql_boxes);
        dbDelta($sql_transactions);
        
        // Store database version for future upgrades
        update_option('sl_payments_db_version', SL_PAYMENTS_VERSION);
    }
    
    /**
     * Set default plugin options
     */
    private static function set_default_options() {
        // Default settings
        // Note: Fee is handled entirely by the LSL script, not WordPress
        $defaults = array(
            'payment_timeout' => 900, // 15 minutes
            'exchange_rate' => 250, // L$ per $1 USD (approximate)
            'use_live_rate' => false,
            'debug_mode' => false,
        );
        
        foreach ($defaults as $key => $value) {
            if (get_option('sl_payments_' . $key) === false) {
                add_option('sl_payments_' . $key, $value);
            }
        }
    }
    
    /**
     * Generate a new API key
     */
    public static function generate_api_key() {
        return wp_generate_password(32, false, false);
    }
    
    /**
     * Generate a unique transaction ID
     */
    public static function generate_transaction_id() {
        return 'sl_' . bin2hex(random_bytes(16));
    }
}
