<?php
/**
 * Admin settings and management pages
 */

if (!defined('WPINC')) {
    die;
}

class SL_Admin {
    
    /**
     * Initialize admin functionality
     */
    public static function init() {
        add_action('admin_menu', array(__CLASS__, 'add_menu_pages'));
        add_action('admin_init', array(__CLASS__, 'register_settings'));
        add_action('wp_ajax_sl_generate_api_key', array(__CLASS__, 'ajax_generate_api_key'));
        add_action('wp_ajax_sl_delete_box', array(__CLASS__, 'ajax_delete_box'));
    }
    
    /**
     * Add admin menu pages
     */
    public static function add_menu_pages() {
        add_menu_page(
            __('DBG Payments', 'secondlife-payments'),
            __('DBG Payments', 'secondlife-payments'),
            'manage_options',
            'sl-payments',
            array(__CLASS__, 'render_settings_page'),
            'dashicons-money-alt',
            56
        );
        
        add_submenu_page(
            'sl-payments',
            __('Settings', 'secondlife-payments'),
            __('Settings', 'secondlife-payments'),
            'manage_options',
            'sl-payments',
            array(__CLASS__, 'render_settings_page')
        );
        
        add_submenu_page(
            'sl-payments',
            __('Payment Boxes', 'secondlife-payments'),
            __('Payment Boxes', 'secondlife-payments'),
            'manage_options',
            'sl-payments-boxes',
            array(__CLASS__, 'render_boxes_page')
        );
        
        add_submenu_page(
            'sl-payments',
            __('Transactions', 'secondlife-payments'),
            __('Transactions', 'secondlife-payments'),
            'manage_options',
            'sl-payments-transactions',
            array(__CLASS__, 'render_transactions_page')
        );
    }
    
    /**
     * Register plugin settings
     */
    public static function register_settings() {
        // General settings section
        add_settings_section(
            'sl_payments_general',
            __('General Settings', 'secondlife-payments'),
            null,
            'sl-payments'
        );
        
        // Payment timeout
        register_setting('sl-payments', 'sl_payments_payment_timeout', array(
            'type' => 'integer',
            'sanitize_callback' => 'absint',
            'default' => 900,
        ));
        add_settings_field(
            'sl_payments_payment_timeout',
            __('Payment Timeout (seconds)', 'secondlife-payments'),
            array(__CLASS__, 'render_number_field'),
            'sl-payments',
            'sl_payments_general',
            array(
                'name' => 'sl_payments_payment_timeout',
                'description' => __('How long to wait for payment before expiring (default: 900 = 15 minutes)', 'secondlife-payments'),
                'min' => 60,
                'max' => 3600,
            )
        );
        
        // Use live exchange rate
        register_setting('sl-payments', 'sl_payments_use_live_rate', array(
            'type' => 'boolean',
            'sanitize_callback' => function($val) {
                return (bool) $val;
            },
            'default' => false,
        ));
        add_settings_field(
            'sl_payments_use_live_rate',
            __('Use Live Exchange Rate', 'secondlife-payments'),
            array(__CLASS__, 'render_live_rate_field'),
            'sl-payments',
            'sl_payments_general',
            array(
                'name' => 'sl_payments_use_live_rate',
            )
        );
        
        // Exchange rate (fallback/manual)
        register_setting('sl-payments', 'sl_payments_exchange_rate', array(
            'type' => 'integer',
            'sanitize_callback' => 'absint',
            'default' => 250,
        ));
        add_settings_field(
            'sl_payments_exchange_rate',
            __('Fallback Exchange Rate', 'secondlife-payments'),
            array(__CLASS__, 'render_number_field'),
            'sl-payments',
            'sl_payments_general',
            array(
                'name' => 'sl_payments_exchange_rate',
                'description' => __('L$ per $1 USD - used when live rate is disabled or unavailable', 'secondlife-payments'),
                'min' => 1,
            )
        );
        
        // Default box API key
        register_setting('sl-payments', 'sl_payments_default_box_key', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
        ));
        add_settings_field(
            'sl_payments_default_box_key',
            __('Payment Box', 'secondlife-payments'),
            array(__CLASS__, 'render_box_select'),
            'sl-payments',
            'sl_payments_general',
            array(
                'name' => 'sl_payments_default_box_key',
                'description' => __('Select which payment box to use for checkout', 'secondlife-payments'),
            )
        );
        
        // Round-robin mode
        register_setting('sl-payments', 'sl_payments_round_robin', array(
            'type' => 'boolean',
            'sanitize_callback' => function($val) {
                return (bool) $val;
            },
            'default' => false,
        ));
        add_settings_field(
            'sl_payments_round_robin',
            __('Round-Robin Distribution', 'secondlife-payments'),
            array(__CLASS__, 'render_round_robin_field'),
            'sl-payments',
            'sl_payments_general',
            array(
                'name' => 'sl_payments_round_robin',
            )
        );
        
        // Box SLURL override
        register_setting('sl-payments', 'sl_payments_box_slurl', array(
            'type' => 'string',
            'sanitize_callback' => 'esc_url_raw',
        ));
        add_settings_field(
            'sl_payments_box_slurl',
            __('Payment Box SLURL (optional)', 'secondlife-payments'),
            array(__CLASS__, 'render_text_field'),
            'sl-payments',
            'sl_payments_general',
            array(
                'name' => 'sl_payments_box_slurl',
                'description' => __('Override the SLURL shown to customers. Leave blank to auto-detect from box. Note: When round-robin is enabled, each box uses its own location.', 'secondlife-payments'),
                'placeholder' => 'secondlife://RegionName/128/128/25',
            )
        );
        
        // Debug mode
        register_setting('sl-payments', 'sl_payments_debug_mode', array(
            'type' => 'boolean',
            'sanitize_callback' => function($val) {
                return (bool) $val;
            },
            'default' => false,
        ));
        add_settings_field(
            'sl_payments_debug_mode',
            __('Debug Mode', 'secondlife-payments'),
            array(__CLASS__, 'render_checkbox_field'),
            'sl-payments',
            'sl_payments_general',
            array(
                'name' => 'sl_payments_debug_mode',
                'description' => __('Enable debug logging', 'secondlife-payments'),
            )
        );
        
        // Security settings section
        add_settings_section(
            'sl_payments_security',
            __('Security Settings', 'secondlife-payments'),
            array(__CLASS__, 'render_security_section_description'),
            'sl-payments'
        );
        
        // Verify Second Life origin (FCrDNS)
        register_setting('sl-payments', 'sl_payments_verify_sl_origin', array(
            'type' => 'boolean',
            'sanitize_callback' => function($val) {
                return (bool) $val;
            },
            'default' => true,
        ));
        add_settings_field(
            'sl_payments_verify_sl_origin',
            __('Verify Second Life Origin', 'secondlife-payments'),
            array(__CLASS__, 'render_checkbox_field'),
            'sl-payments',
            'sl_payments_security',
            array(
                'name' => 'sl_payments_verify_sl_origin',
                'description' => __('Use FCrDNS to verify requests come from Second Life servers (recommended)', 'secondlife-payments'),
                'default' => true,
            )
        );
        
        // Timestamp tolerance
        register_setting('sl-payments', 'sl_payments_timestamp_tolerance', array(
            'type' => 'integer',
            'sanitize_callback' => 'absint',
            'default' => 300,
        ));
        add_settings_field(
            'sl_payments_timestamp_tolerance',
            __('Timestamp Tolerance (seconds)', 'secondlife-payments'),
            array(__CLASS__, 'render_number_field'),
            'sl-payments',
            'sl_payments_security',
            array(
                'name' => 'sl_payments_timestamp_tolerance',
                'description' => __('Maximum allowed time difference for API requests (default: 300 = 5 minutes). Prevents replay attacks.', 'secondlife-payments'),
                'min' => 60,
                'max' => 900,
                'default' => 300,
            )
        );
    }
    
    /**
     * Render security section description
     */
    public static function render_security_section_description() {
        ?>
        <p class="description">
            <?php _e('These settings help protect your payment API from unauthorized access.', 'secondlife-payments'); ?>
        </p>
        <div style="padding: 10px; background: #fff8e5; border-left: 4px solid #dba617; margin-bottom: 15px;">
            <strong><?php _e('About FCrDNS Verification:', 'secondlife-payments'); ?></strong>
            <p style="margin: 5px 0 0 0;">
                <?php _e('Forward-Confirmed Reverse DNS verifies that API requests actually originate from Linden Lab\'s Second Life servers. This prevents attackers from spoofing requests even if they obtain an API key. Disable only if you experience connectivity issues.', 'secondlife-payments'); ?>
            </p>
        </div>
        <?php
    }
    
    /**
     * Render text field
     */
    public static function render_text_field($args) {
        $value = get_option($args['name'], '');
        ?>
        <input type="text" 
               name="<?php echo esc_attr($args['name']); ?>" 
               value="<?php echo esc_attr($value); ?>"
               placeholder="<?php echo esc_attr($args['placeholder'] ?? ''); ?>"
               class="regular-text">
        <?php if (!empty($args['description'])): ?>
            <p class="description"><?php echo esc_html($args['description']); ?></p>
        <?php endif;
    }
    
    /**
     * Render number field
     */
    public static function render_number_field($args) {
        $default = $args['default'] ?? '';
        $value = get_option($args['name'], $default);
        // If option exists but is empty, use default
        if ($value === '' && $default !== '') {
            $value = $default;
        }
        ?>
        <input type="number" 
               name="<?php echo esc_attr($args['name']); ?>" 
               value="<?php echo esc_attr($value); ?>"
               min="<?php echo esc_attr($args['min'] ?? ''); ?>"
               max="<?php echo esc_attr($args['max'] ?? ''); ?>"
               step="<?php echo esc_attr($args['step'] ?? 1); ?>"
               class="small-text">
        <?php if (!empty($args['description'])): ?>
            <p class="description"><?php echo esc_html($args['description']); ?></p>
        <?php endif;
    }
    
    /**
     * Render checkbox field
     */
    public static function render_checkbox_field($args) {
        $default = $args['default'] ?? false;
        // For checkboxes, we need special handling:
        // - If option doesn't exist at all, use the default
        // - If option exists (even as false/0), use that value
        $option_exists = get_option($args['name'], 'NOT_SET') !== 'NOT_SET';
        if ($option_exists) {
            $value = (bool) get_option($args['name'], false);
        } else {
            $value = $default;
        }
        ?>
        <input type="checkbox" 
               name="<?php echo esc_attr($args['name']); ?>" 
               value="1"
               <?php checked($value, true); ?>>
        <?php if (!empty($args['description'])): ?>
            <span class="description"><?php echo esc_html($args['description']); ?></span>
        <?php endif;
    }
    
    /**
     * Render round-robin distribution field with status
     */
    public static function render_round_robin_field($args) {
        global $wpdb;
        $table = $wpdb->prefix . 'sl_boxes';
        
        $enabled = get_option($args['name'], false);
        
        // Get count of active boxes
        $total_boxes = $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE is_active = 1");
        
        // Get count of online boxes (seen in last 2 minutes)
        $online_threshold = gmdate('Y-m-d H:i:s', time() - 120);
        $online_boxes = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table WHERE is_active = 1 AND last_seen > %s",
            $online_threshold
        ));
        
        // Get all active boxes for display
        $boxes = $wpdb->get_results(
            "SELECT id, owner_name, region, last_seen FROM $table WHERE is_active = 1 ORDER BY id ASC"
        );
        ?>
        <input type="checkbox" 
               name="<?php echo esc_attr($args['name']); ?>" 
               value="1"
               <?php checked($enabled, true); ?>>
        <span class="description">
            <?php _e('Distribute orders across multiple payment boxes', 'secondlife-payments'); ?>
        </span>
        
        <div style="margin-top: 10px; padding: 15px; background: #f0f6fc; border: 1px solid #c3c4c7; border-radius: 4px;">
            <strong><?php _e('Payment Box Pool:', 'secondlife-payments'); ?></strong>
            <span style="margin-left: 10px;">
                <?php printf(
                    __('%d active, %d online', 'secondlife-payments'),
                    $total_boxes,
                    $online_boxes
                ); ?>
            </span>
            
            <?php if ($enabled): ?>
                <span style="color: #00a32a; margin-left: 10px;">● <?php _e('Round-Robin Active', 'secondlife-payments'); ?></span>
            <?php else: ?>
                <span style="color: #dba617; margin-left: 10px;">● <?php _e('Single Box Mode', 'secondlife-payments'); ?></span>
            <?php endif; ?>
            
            <?php if ($total_boxes > 1): ?>
                <table style="margin-top: 10px; width: 100%; font-size: 13px;">
                    <thead>
                        <tr style="text-align: left;">
                            <th style="padding: 5px;"><?php _e('ID', 'secondlife-payments'); ?></th>
                            <th style="padding: 5px;"><?php _e('Owner', 'secondlife-payments'); ?></th>
                            <th style="padding: 5px;"><?php _e('Region', 'secondlife-payments'); ?></th>
                            <th style="padding: 5px;"><?php _e('Status', 'secondlife-payments'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($boxes as $box): 
                            $last_seen_ts = $box->last_seen ? strtotime($box->last_seen . ' UTC') : 0;
                            $is_online = $last_seen_ts > (time() - 120);
                        ?>
                            <tr>
                                <td style="padding: 5px;"><?php echo esc_html($box->id); ?></td>
                                <td style="padding: 5px;"><?php echo esc_html($box->owner_name ?: 'Unknown'); ?></td>
                                <td style="padding: 5px;"><?php echo esc_html($box->region ?: 'Unknown'); ?></td>
                                <td style="padding: 5px;">
                                    <?php if ($is_online): ?>
                                        <span style="color: #00a32a;">● <?php _e('Online', 'secondlife-payments'); ?></span>
                                    <?php else: ?>
                                        <span style="color: #d63638;">● <?php _e('Offline', 'secondlife-payments'); ?></span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php elseif ($total_boxes == 1): ?>
                <p style="margin: 10px 0 0 0; color: #666;">
                    <?php _e('Only one box registered. Add more boxes to enable distribution.', 'secondlife-payments'); ?>
                </p>
            <?php else: ?>
                <p style="margin: 10px 0 0 0; color: #d63638;">
                    <?php _e('No boxes registered yet.', 'secondlife-payments'); ?>
                </p>
            <?php endif; ?>
            
            <?php if ($enabled && $total_boxes > 1): ?>
                <p style="margin: 10px 0 0 0; font-style: italic; color: #666;">
                    <?php _e('Orders will be distributed to online boxes. Each customer will be directed to a specific box location.', 'secondlife-payments'); ?>
                </p>
            <?php endif; ?>
        </div>
        <?php
    }
    
    /**
     * Render live exchange rate field with current rate display
     */
    public static function render_live_rate_field($args) {
        $use_live = get_option($args['name'], false);
        $current_rate = self::get_exchange_rate();
        $live_rate = self::get_live_exchange_rate();
        ?>
        <input type="checkbox" 
               name="<?php echo esc_attr($args['name']); ?>" 
               value="1"
               <?php checked($use_live, true); ?>>
        <span class="description">
            <?php _e('Fetch live L$/USD rate from Linden Lab API', 'secondlife-payments'); ?>
        </span>
        
        <div style="margin-top: 10px; padding: 10px; background: #f0f0f1; border-left: 4px solid #2271b1;">
            <strong><?php _e('Current Exchange Rate:', 'secondlife-payments'); ?></strong>
            <span style="font-size: 16px; margin-left: 10px;">
                L$<?php echo number_format($current_rate, 2); ?> = $1.00 USD
            </span>
            <?php if ($use_live): ?>
                <span style="color: #00a32a; margin-left: 10px;">● <?php _e('Live', 'secondlife-payments'); ?></span>
            <?php else: ?>
                <span style="color: #dba617; margin-left: 10px;">● <?php _e('Manual', 'secondlife-payments'); ?></span>
            <?php endif; ?>
            
            <?php if ($live_rate !== false): ?>
                <br><small style="color: #666;">
                    <?php printf(__('Live rate from Linden Lab: L$%s per $1 USD', 'secondlife-payments'), number_format($live_rate, 2)); ?>
                </small>
            <?php else: ?>
                <br><small style="color: #d63638;">
                    <?php _e('Unable to fetch live rate - using fallback', 'secondlife-payments'); ?>
                </small>
            <?php endif; ?>
        </div>
        <?php
    }
    
    /**
     * Render box select dropdown with inline diagnostic info
     */
    public static function render_box_select($args) {
        global $wpdb;
        $table = $wpdb->prefix . 'sl_boxes';
        $boxes = $wpdb->get_results("SELECT * FROM $table WHERE is_active = 1");
        $value = get_option($args['name'], '');
        ?>
        <select name="<?php echo esc_attr($args['name']); ?>">
            <option value=""><?php _e('-- Select a box --', 'secondlife-payments'); ?></option>
            <?php foreach ($boxes as $box): ?>
                <option value="<?php echo esc_attr($box->api_key); ?>" <?php selected($value, $box->api_key); ?>>
                    <?php echo esc_html($box->owner_name ?: 'Unknown') . ' @ ' . esc_html($box->region ?: 'Unknown region') . ' (ID: ' . $box->id . ')'; ?>
                </option>
            <?php endforeach; ?>
        </select>
        <?php if (!empty($args['description'])): ?>
            <p class="description"><?php echo esc_html($args['description']); ?></p>
        <?php endif;
        
        if (empty($boxes)): ?>
            <p class="description" style="color: #d63638;">
                <?php _e('No payment boxes registered yet. Create an API key below and configure it in your Second Life payment box.', 'secondlife-payments'); ?>
            </p>
        <?php else:
            // Show selected box info inline
            $selected_box = null;
            foreach ($boxes as $box) {
                if ($box->api_key === $value) {
                    $selected_box = $box;
                    break;
                }
            }
            
            if ($selected_box): 
                $last_seen_ts = $selected_box->last_seen ? strtotime($selected_box->last_seen . ' UTC') : 0;
                $is_online = $last_seen_ts > (time() - 120);
            ?>
                <div style="margin-top: 15px; padding: 15px; background: #f0f6fc; border: 1px solid #c3c4c7; border-radius: 4px;">
                    <strong><?php _e('Selected Box Details:', 'secondlife-payments'); ?></strong>
                    <table style="margin-top: 10px; width: 100%;">
                        <tr><td style="width: 120px;"><strong>Box ID:</strong></td><td><?php echo esc_html($selected_box->id); ?></td></tr>
                        <tr><td><strong>Owner:</strong></td><td><?php echo esc_html($selected_box->owner_name ?: 'Not registered'); ?></td></tr>
                        <tr><td><strong>Region:</strong></td><td><?php echo esc_html($selected_box->region ?: 'Unknown'); ?></td></tr>
                        <tr><td><strong>Position:</strong></td><td><?php echo esc_html($selected_box->position ?: 'Unknown'); ?></td></tr>
                        <tr>
                            <td><strong>Status:</strong></td>
                            <td>
                                <?php if ($is_online): ?>
                                    <span style="color: #00a32a;">● Online</span>
                                <?php else: ?>
                                    <span style="color: #d63638;">● Offline</span>
                                <?php endif; ?>
                                (Last seen: <?php echo $selected_box->last_seen ? esc_html($selected_box->last_seen . ' UTC') : 'Never'; ?>)
                            </td>
                        </tr>
                        <tr><td><strong>Server Time:</strong></td><td><?php echo esc_html(gmdate('Y-m-d H:i:s')); ?> UTC</td></tr>
                    </table>
                </div>
            <?php endif;
        endif;
    }
    
    /**
     * Render settings page
     */
    public static function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Check configuration status
        $issues = self::check_configuration();
        
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <?php if (!empty($issues)): ?>
                <div class="notice notice-warning">
                    <p><strong><?php _e('Configuration Issues:', 'secondlife-payments'); ?></strong></p>
                    <ul style="list-style: disc; margin-left: 20px;">
                        <?php foreach ($issues as $issue): ?>
                            <li><?php echo wp_kses_post($issue); ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php else: ?>
                <div class="notice notice-success">
                    <p>✓ <?php _e('DBG Payments is configured and ready to accept payments.', 'secondlife-payments'); ?></p>
                </div>
            <?php endif; ?>
            
            
            <form action="options.php" method="post">
                <?php
                settings_fields('sl-payments');
                do_settings_sections('sl-payments');
                submit_button();
                ?>
            </form>
            
            <hr>
            
            <h2><?php _e('Generate New API Key', 'secondlife-payments'); ?></h2>
            <p><?php _e('Create a new API key to configure a payment box in Second Life.', 'secondlife-payments'); ?></p>
            
            <div id="sl-api-key-generator">
                <button type="button" class="button button-secondary" id="sl-generate-key">
                    <?php _e('Generate API Key', 'secondlife-payments'); ?>
                </button>
                <div id="sl-new-key-display" style="display: none; margin-top: 10px; padding: 15px; background: #f0f0f1; border-left: 4px solid #2271b1;">
                    <strong><?php _e('New API Key:', 'secondlife-payments'); ?></strong>
                    <code id="sl-new-key-value" style="font-size: 14px; padding: 5px; margin-left: 10px;"></code>
                    <p class="description" style="margin-top: 10px;">
                        <?php _e('Copy this key and add it to your payment box config notecard. This key will only be shown once.', 'secondlife-payments'); ?>
                    </p>
                </div>
            </div>
            
            <script>
            jQuery(document).ready(function($) {
                $('#sl-generate-key').on('click', function() {
                    var $btn = $(this);
                    $btn.prop('disabled', true).text('<?php _e('Generating...', 'secondlife-payments'); ?>');
                    
                    $.post(ajaxurl, {
                        action: 'sl_generate_api_key',
                        _wpnonce: '<?php echo wp_create_nonce('sl_generate_key'); ?>'
                    }, function(response) {
                        if (response.success) {
                            $('#sl-new-key-value').text(response.data.api_key);
                            $('#sl-new-key-display').slideDown();
                        } else {
                            alert(response.data || 'Error generating key');
                        }
                        $btn.prop('disabled', false).text('<?php _e('Generate Another Key', 'secondlife-payments'); ?>');
                    });
                });
                
            });
            </script>
        </div>
        <?php
    }
    
    /**
     * Render boxes management page
     */
    public static function render_boxes_page() {
        global $wpdb;
        $table = $wpdb->prefix . 'sl_boxes';
        $boxes = $wpdb->get_results("SELECT * FROM $table ORDER BY created_at DESC");
        ?>
        <div class="wrap">
            <h1><?php _e('Payment Boxes', 'secondlife-payments'); ?></h1>
            
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th style="width: 60px;"><?php _e('Box ID', 'secondlife-payments'); ?></th>
                        <th><?php _e('Owner', 'secondlife-payments'); ?></th>
                        <th><?php _e('Region', 'secondlife-payments'); ?></th>
                        <th><?php _e('API Key', 'secondlife-payments'); ?></th>
                        <th><?php _e('Status', 'secondlife-payments'); ?></th>
                        <th><?php _e('Last Seen', 'secondlife-payments'); ?></th>
                        <th><?php _e('Actions', 'secondlife-payments'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($boxes)): ?>
                        <tr>
                            <td colspan="7"><?php _e('No payment boxes registered yet.', 'secondlife-payments'); ?></td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($boxes as $box): 
                            // Compare using GMT timestamps for consistency
                            $last_seen_ts = $box->last_seen ? strtotime($box->last_seen . ' UTC') : 0;
                            $is_online = $last_seen_ts > (time() - 120);
                        ?>
                            <tr>
                                <td>
                                    <strong><?php echo esc_html($box->id); ?></strong>
                                </td>
                                <td>
                                    <strong><?php echo esc_html($box->owner_name ?: 'Unknown'); ?></strong>
                                    <br><small><?php echo esc_html($box->owner_key); ?></small>
                                </td>
                                <td>
                                    <?php echo esc_html($box->region ?: '-'); ?>
                                    <?php if ($box->position): ?>
                                        <br><small><?php echo esc_html($box->position); ?></small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <code><?php echo esc_html(substr($box->api_key, 0, 8) . '...'); ?></code>
                                </td>
                                <td>
                                    <?php if ($box->is_active): ?>
                                        <?php if ($is_online): ?>
                                            <span style="color: #00a32a;">● <?php _e('Online', 'secondlife-payments'); ?></span>
                                        <?php else: ?>
                                            <span style="color: #dba617;">● <?php _e('Offline', 'secondlife-payments'); ?></span>
                                        <?php endif; ?>
                                    <?php else: ?>
                                        <span style="color: #d63638;">● <?php _e('Disabled', 'secondlife-payments'); ?></span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php echo $box->last_seen ? esc_html(human_time_diff(strtotime($box->last_seen))) . ' ago' : '-'; ?>
                                </td>
                                <td>
                                    <button type="button" class="button button-small button-link-delete sl-delete-box" 
                                            data-id="<?php echo esc_attr($box->id); ?>">
                                        <?php _e('Delete', 'secondlife-payments'); ?>
                                    </button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            $('.sl-delete-box').on('click', function() {
                if (!confirm('<?php _e('Are you sure you want to delete this box?', 'secondlife-payments'); ?>')) {
                    return;
                }
                
                var $btn = $(this);
                var boxId = $btn.data('id');
                
                $.post(ajaxurl, {
                    action: 'sl_delete_box',
                    box_id: boxId,
                    _wpnonce: '<?php echo wp_create_nonce('sl_delete_box'); ?>'
                }, function(response) {
                    if (response.success) {
                        $btn.closest('tr').fadeOut(function() { $(this).remove(); });
                    } else {
                        alert(response.data || 'Error deleting box');
                    }
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * Render transactions page
     */
    public static function render_transactions_page() {
        global $wpdb;
        $table = $wpdb->prefix . 'sl_transactions';
        
        // Handle purge pending action
        if (isset($_POST['sl_purge_pending']) && wp_verify_nonce($_POST['_wpnonce'], 'sl_purge_pending')) {
            $deleted = $wpdb->query("DELETE FROM $table WHERE status IN ('pending', 'expired')");
            echo '<div class="notice notice-success"><p>' . sprintf(__('Purged %d pending/expired transactions.', 'secondlife-payments'), $deleted) . '</p></div>';
        }
        
        // Handle purge all action
        if (isset($_POST['sl_purge_all']) && wp_verify_nonce($_POST['_wpnonce'], 'sl_purge_all')) {
            $deleted = $wpdb->query("DELETE FROM $table");
            echo '<div class="notice notice-success"><p>' . sprintf(__('Deleted all %d transactions.', 'secondlife-payments'), $deleted) . '</p></div>';
        }
        
        // Handle delete single transaction
        if (isset($_GET['delete_txn']) && wp_verify_nonce($_GET['_wpnonce'], 'sl_delete_txn')) {
            $txn_id = sanitize_text_field($_GET['delete_txn']);
            $deleted = $wpdb->delete($table, array('transaction_id' => $txn_id));
            if ($deleted) {
                echo '<div class="notice notice-success"><p>' . __('Transaction deleted.', 'secondlife-payments') . '</p></div>';
            }
        }
        
        $page = max(1, intval($_GET['paged'] ?? 1));
        $per_page = 20;
        $offset = ($page - 1) * $per_page;
        
        $total = $wpdb->get_var("SELECT COUNT(*) FROM $table");
        $pending_count = $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE status IN ('pending', 'expired')");
        $transactions = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table ORDER BY created_at DESC LIMIT %d OFFSET %d",
            $per_page, $offset
        ));
        
        $total_pages = ceil($total / $per_page);
        ?>
        <div class="wrap">
            <h1><?php _e('Transactions', 'secondlife-payments'); ?></h1>
            
            <div style="margin-bottom: 15px; padding: 10px; background: #f0f0f1; border-left: 4px solid #2271b1;">
                <strong><?php _e('Transaction Management', 'secondlife-payments'); ?></strong>
                <p style="margin: 5px 0;"><?php printf(__('Total: %d | Pending/Expired: %d', 'secondlife-payments'), $total, $pending_count); ?></p>
                <form method="post" style="display: inline;">
                    <?php wp_nonce_field('sl_purge_pending'); ?>
                    <button type="submit" name="sl_purge_pending" class="button button-secondary" 
                            <?php echo $pending_count == 0 ? 'disabled' : ''; ?>
                            onclick="return confirm('Delete all <?php echo $pending_count; ?> pending/expired transactions? This cannot be undone.');">
                        <?php _e('Purge Pending/Expired', 'secondlife-payments'); ?>
                    </button>
                </form>
                <form method="post" style="display: inline; margin-left: 10px;">
                    <?php wp_nonce_field('sl_purge_all'); ?>
                    <button type="submit" name="sl_purge_all" class="button button-link-delete" 
                            <?php echo $total == 0 ? 'disabled' : ''; ?>
                            onclick="return confirm('DELETE ALL <?php echo $total; ?> TRANSACTIONS? This cannot be undone!');">
                        <?php _e('Delete All Transactions', 'secondlife-payments'); ?>
                    </button>
                </form>
            </div>
            
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th><?php _e('Transaction ID', 'secondlife-payments'); ?></th>
                        <th><?php _e('Box ID', 'secondlife-payments'); ?></th>
                        <th><?php _e('Buyer', 'secondlife-payments'); ?></th>
                        <th><?php _e('Amount', 'secondlife-payments'); ?></th>
                        <th><?php _e('Fee', 'secondlife-payments'); ?></th>
                        <th><?php _e('Status', 'secondlife-payments'); ?></th>
                        <th><?php _e('Order', 'secondlife-payments'); ?></th>
                        <th><?php _e('Date', 'secondlife-payments'); ?></th>
                        <th><?php _e('Actions', 'secondlife-payments'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($transactions)): ?>
                        <tr>
                            <td colspan="9"><?php _e('No transactions yet.', 'secondlife-payments'); ?></td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($transactions as $txn): ?>
                            <tr>
                                <td>
                                    <code><?php echo esc_html(substr($txn->transaction_id, 0, 16) . '...'); ?></code>
                                </td>
                                <td><?php echo esc_html($txn->box_id); ?></td>
                                <td><?php echo esc_html($txn->buyer_sl_name); ?></td>
                                <td>L$<?php echo number_format($txn->amount); ?></td>
                                <td>L$<?php echo number_format($txn->fee); ?></td>
                                <td>
                                    <?php 
                                    $status_colors = array(
                                        'pending' => '#dba617',
                                        'completed' => '#00a32a',
                                        'expired' => '#d63638',
                                        'failed' => '#d63638',
                                    );
                                    $color = $status_colors[$txn->status] ?? '#666';
                                    
                                    // Check if actually expired
                                    $now = gmdate('Y-m-d H:i:s');
                                    $is_expired = $txn->expires_at && $txn->expires_at < $now;
                                    ?>
                                    <span style="color: <?php echo $color; ?>;">
                                        <?php echo esc_html(ucfirst($txn->status)); ?>
                                    </span>
                                    <?php if ($txn->status === 'pending' && $is_expired): ?>
                                        <br><small style="color: #d63638;">(Actually expired)</small>
                                    <?php endif; ?>
                                    <?php if ($txn->status === 'pending' && $txn->expires_at): ?>
                                        <br><small>Expires: <?php echo esc_html($txn->expires_at); ?> UTC</small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($txn->order_id): ?>
                                        <a href="<?php echo admin_url('post.php?post=' . $txn->order_id . '&action=edit'); ?>">
                                            #<?php echo $txn->order_id; ?>
                                        </a>
                                    <?php else: ?>
                                        -
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php echo esc_html(date('M j, Y g:i a', strtotime($txn->created_at))); ?>
                                </td>
                                <td>
                                    <?php if ($txn->status !== 'completed'): ?>
                                        <a href="<?php echo wp_nonce_url(add_query_arg('delete_txn', $txn->transaction_id), 'sl_delete_txn'); ?>" 
                                           onclick="return confirm('Delete this transaction?');"
                                           style="color: #d63638;">
                                            <?php _e('Delete', 'secondlife-payments'); ?>
                                        </a>
                                    <?php else: ?>
                                        -
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
            
            <?php if ($total_pages > 1): ?>
                <div class="tablenav bottom">
                    <div class="tablenav-pages">
                        <?php
                        echo paginate_links(array(
                            'base' => add_query_arg('paged', '%#%'),
                            'format' => '',
                            'current' => $page,
                            'total' => $total_pages,
                        ));
                        ?>
                    </div>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }
    
    /**
     * AJAX: Generate new API key
     */
    public static function ajax_generate_api_key() {
        check_ajax_referer('sl_generate_key');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'sl_boxes';
        
        require_once SL_PAYMENTS_PATH . 'includes/class-sl-activator.php';
        $api_key = SL_Activator::generate_api_key();
        
        // Create placeholder box entry
        $wpdb->insert($table, array(
            'api_key' => $api_key,
            'owner_key' => '',
            'is_active' => 1,
        ));
        
        wp_send_json_success(array('api_key' => $api_key));
    }
    
    /**
     * AJAX: Delete box
     */
    public static function ajax_delete_box() {
        check_ajax_referer('sl_delete_box');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'sl_boxes';
        
        $box_id = intval($_POST['box_id'] ?? 0);
        if (!$box_id) {
            wp_send_json_error('Invalid box ID');
        }
        
        $wpdb->delete($table, array('id' => $box_id));
        
        wp_send_json_success();
    }
    
    /**
     * Check if the plugin is properly configured
     * 
     * @return array List of configuration issues (empty if all good)
     */
    public static function check_configuration() {
        $issues = array();
        
        // Check if WooCommerce is active
        if (!class_exists('WooCommerce')) {
            $issues[] = __('WooCommerce is not installed or activated.', 'secondlife-payments');
        }
        
        // Check if gateway is enabled
        $gateway_settings = get_option('woocommerce_secondlife_settings', array());
        if (empty($gateway_settings['enabled']) || $gateway_settings['enabled'] !== 'yes') {
            $issues[] = sprintf(
                __('Payment gateway is not enabled. <a href="%s">Enable it in WooCommerce settings</a>.', 'secondlife-payments'),
                admin_url('admin.php?page=wc-settings&tab=checkout&section=secondlife')
            );
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'sl_boxes';
        
        // Check if round-robin mode is enabled
        $round_robin = get_option('sl_payments_round_robin', false);
        
        if ($round_robin) {
            // Round-robin mode: check for ANY online boxes
            $online_threshold = gmdate('Y-m-d H:i:s', time() - 300);
            $online_count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $table WHERE is_active = 1 AND last_seen > %s",
                $online_threshold
            ));
            
            $total_count = $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE is_active = 1");
            
            if ($total_count == 0) {
                $issues[] = __('Round-robin is enabled but no payment boxes are registered. Generate API keys and set up boxes in Second Life.', 'secondlife-payments');
            } elseif ($online_count == 0) {
                $issues[] = sprintf(
                    __('Round-robin is enabled but all %d payment box(es) appear offline. Make sure at least one box is rezzed and running in Second Life.', 'secondlife-payments'),
                    $total_count
                );
            } elseif ($online_count < $total_count) {
                // Warning, not blocking - some boxes are offline
                $issues[] = sprintf(
                    __('Round-robin: %d of %d boxes online. Offline boxes will be skipped.', 'secondlife-payments'),
                    $online_count,
                    $total_count
                );
            }
        } else {
            // Single box mode: check the selected box
            $box_key = get_option('sl_payments_default_box_key', '');
            if (empty($box_key)) {
                $issues[] = __('No payment box selected. Generate an API key below and set up a box in Second Life, or select an existing box.', 'secondlife-payments');
            } else {
                $box = $wpdb->get_row($wpdb->prepare(
                    "SELECT is_active, last_seen FROM $table WHERE api_key = %s",
                    $box_key
                ));
                
                if (!$box) {
                    $issues[] = __('Selected payment box not found in database. It may have been deleted.', 'secondlife-payments');
                } elseif (!$box->is_active) {
                    $issues[] = __('Selected payment box is disabled.', 'secondlife-payments');
                } else {
                    $last_seen_ts = strtotime($box->last_seen . ' UTC');
                    if (empty($box->last_seen) || $last_seen_ts < time() - 300) {
                        $issues[] = __('Selected payment box appears to be offline (not seen in last 5 minutes). Make sure the box is rezzed and the script is running in Second Life.', 'secondlife-payments');
                    }
                }
            }
        }
        
        return $issues;
    }
    
    /**
     * Fetch live exchange rate from Linden Lab API
     * 
     * @return float|false Exchange rate (L$ per $1 USD) or false on failure
     */
    public static function get_live_exchange_rate() {
        // Check cache first (cache for 15 minutes)
        $cached = get_transient('sl_payments_live_exchange_rate');
        if ($cached !== false) {
            return floatval($cached);
        }
        
        // Fetch from Linden Lab official API
        $response = wp_remote_get('https://api.secondlife.com/datafeeds/homepage.txt', array(
            'timeout' => 10,
        ));
        
        if (is_wp_error($response)) {
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        if (empty($body)) {
            return false;
        }
        
        // Parse the text feed - look for exchange_rate line
        $lines = explode("\n", $body);
        $next_is_rate = false;
        
        foreach ($lines as $line) {
            $line = trim($line);
            if ($next_is_rate) {
                $rate = floatval($line);
                if ($rate > 0) {
                    // Cache for 15 minutes
                    set_transient('sl_payments_live_exchange_rate', $rate, 15 * MINUTE_IN_SECONDS);
                    return $rate;
                }
                break;
            }
            if ($line === 'exchange_rate') {
                $next_is_rate = true;
            }
        }
        
        return false;
    }
    
    /**
     * Get the current exchange rate (live or manual)
     * 
     * @return float Exchange rate (L$ per $1 USD)
     */
    public static function get_exchange_rate() {
        $use_live = get_option('sl_payments_use_live_rate', false);
        
        if ($use_live) {
            $live_rate = self::get_live_exchange_rate();
            if ($live_rate !== false) {
                return $live_rate;
            }
        }
        
        // Fall back to manual rate
        return floatval(get_option('sl_payments_exchange_rate', 250));
    }
    
    /**
     * Convert an amount from any currency to USD
     * Uses the Frankfurter API (European Central Bank rates)
     * 
     * @param float $amount The amount to convert
     * @param string $from_currency The source currency code (e.g., 'EUR', 'GBP')
     * @return float|false The amount in USD, or false on failure
     */
    public static function convert_to_usd($amount, $from_currency) {
        $from_currency = strtoupper($from_currency);
        
        // Already USD, no conversion needed
        if ($from_currency === 'USD') {
            return floatval($amount);
        }
        
        // Get cached rates
        $rates = self::get_currency_rates();
        
        if ($rates === false || !isset($rates[$from_currency])) {
            if (get_option('sl_payments_debug_mode', false)) {
                error_log("SL Payments: Could not get exchange rate for {$from_currency}");
            }
            return false;
        }
        
        // rates[$from_currency] is how many of that currency = 1 USD
        // So to convert from_currency to USD: amount / rate
        $usd_amount = floatval($amount) / $rates[$from_currency];
        
        return $usd_amount;
    }
    
    /**
     * Convert an amount from any currency to Linden Dollars
     * 
     * @param float $amount The amount to convert
     * @param string $from_currency The source currency code (e.g., 'EUR', 'GBP', 'USD')
     * @return int|false The amount in L$ (rounded up), or false on failure
     */
    public static function convert_to_linden($amount, $from_currency) {
        // First convert to USD
        $usd_amount = self::convert_to_usd($amount, $from_currency);
        
        if ($usd_amount === false) {
            return false;
        }
        
        // Then convert USD to L$
        $exchange_rate = self::get_exchange_rate();
        
        return (int) ceil($usd_amount * $exchange_rate);
    }
    
    /**
     * Get currency exchange rates (to USD)
     * Caches rates for 1 hour
     * 
     * @return array|false Associative array of currency => rate_to_usd, or false on failure
     */
    public static function get_currency_rates() {
        // Check cache first
        $cached = get_transient('sl_payments_currency_rates');
        if ($cached !== false) {
            return $cached;
        }
        
        // Fetch from Frankfurter API (uses European Central Bank data, free, no API key)
        // We fetch rates with USD as base, so we get how many of each currency = 1 USD
        $response = wp_remote_get('https://api.frankfurter.app/latest?from=USD', array(
            'timeout' => 10,
        ));
        
        if (is_wp_error($response)) {
            if (get_option('sl_payments_debug_mode', false)) {
                error_log('SL Payments: Failed to fetch currency rates: ' . $response->get_error_message());
            }
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (empty($data) || !isset($data['rates'])) {
            if (get_option('sl_payments_debug_mode', false)) {
                error_log('SL Payments: Invalid currency rate response');
            }
            return false;
        }
        
        // Add USD itself (1 USD = 1 USD)
        $rates = $data['rates'];
        $rates['USD'] = 1.0;
        
        // Cache for 1 hour
        set_transient('sl_payments_currency_rates', $rates, HOUR_IN_SECONDS);
        
        if (get_option('sl_payments_debug_mode', false)) {
            error_log('SL Payments: Fetched ' . count($rates) . ' currency rates');
        }
        
        return $rates;
    }
    
    /**
     * Get list of supported currencies
     * 
     * @return array Array of currency codes
     */
    public static function get_supported_currencies() {
        $rates = self::get_currency_rates();
        
        if ($rates === false) {
            // Return common currencies as fallback
            return array('USD', 'EUR', 'GBP', 'CAD', 'AUD', 'JPY', 'CHF');
        }
        
        return array_keys($rates);
    }
}
