<?php
/**
 * REST API endpoints for Second Life payment box communication
 */

if (!defined('WPINC')) {
    die;
}

class SL_API {
    
    /**
     * Valid Second Life server hostname patterns for FCrDNS verification
     * Note: Linden Lab migrated to AWS, so SL servers now have AWS hostnames
     */
    private static $valid_sl_hostname_patterns = array(
        // Legacy Linden Lab hostnames (may still be used for some services)
        '/\.agni\.secondlife\.io$/',
        '/\.aditi\.secondlife\.io$/',
        '/\.lindenlab\.com$/',
        '/\.secondlife\.com$/',
        // AWS hostnames (Linden Lab cloud infrastructure)
        // SL simulators run primarily in us-west-2
        '/^ec2-[\d-]+\.us-west-2\.compute\.amazonaws\.com$/',
        // Also allow us-west-1 as backup region
        '/^ec2-[\d-]+\.us-west-1\.compute\.amazonaws\.com$/',
    );
    
    /**
     * Maximum allowed timestamp difference in seconds (5 minutes)
     */
    const TIMESTAMP_TOLERANCE = 300;
    
    /**
     * Initialize API routes
     */
    public static function init() {
        add_action('rest_api_init', array(__CLASS__, 'register_routes'));
    }
    
    /**
     * Register REST API routes
     */
    public static function register_routes() {
        $namespace = 'sl-payments/v1';
        
        // Box registration endpoint
        register_rest_route($namespace, '/register', array(
            'methods' => 'POST',
            'callback' => array(__CLASS__, 'register_box'),
            'permission_callback' => array(__CLASS__, 'check_api_key'),
        ));
        
        // Heartbeat/keep-alive endpoint
        register_rest_route($namespace, '/heartbeat', array(
            'methods' => 'POST',
            'callback' => array(__CLASS__, 'heartbeat'),
            'permission_callback' => array(__CLASS__, 'check_api_key'),
        ));
        
        // Get pending payments for a box
        register_rest_route($namespace, '/pending/(?P<api_key>[a-zA-Z0-9]+)', array(
            'methods' => 'GET',
            'callback' => array(__CLASS__, 'get_pending'),
            'permission_callback' => '__return_true', // API key in URL
        ));
        
        // Payment completion callback from box
        register_rest_route($namespace, '/complete', array(
            'methods' => 'POST',
            'callback' => array(__CLASS__, 'complete_payment'),
            'permission_callback' => array(__CLASS__, 'check_api_key'),
        ));
        
        // Frontend status polling
        register_rest_route($namespace, '/status/(?P<transaction_id>[a-zA-Z0-9_]+)', array(
            'methods' => 'GET',
            'callback' => array(__CLASS__, 'get_status'),
            'permission_callback' => '__return_true',
        ));
        
        // Initiate a new payment (internal use)
        register_rest_route($namespace, '/initiate', array(
            'methods' => 'POST',
            'callback' => array(__CLASS__, 'initiate_payment'),
            'permission_callback' => function() {
                return current_user_can('manage_options') || wp_verify_nonce($_REQUEST['_wpnonce'] ?? '', 'wp_rest');
            },
        ));
        
        // Match a payment to a pending transaction (called by LSL when payment received)
        register_rest_route($namespace, '/match-payment', array(
            'methods' => 'POST',
            'callback' => array(__CLASS__, 'match_payment'),
            'permission_callback' => array(__CLASS__, 'check_api_key'),
        ));
    }
    
    /**
     * Check API key from header or body, plus security validations
     */
    public static function check_api_key($request) {
        // First, perform security checks (DNS verification, timestamp)
        $security_check = self::validate_request_security($request);
        if (is_wp_error($security_check)) {
            // Log the security failure
            if (get_option('sl_payments_debug_mode', false)) {
                error_log('SL Payments Security: ' . $security_check->get_error_message());
            }
            return false;
        }
        
        $api_key = $request->get_header('X-SL-API-Key');
        
        if (empty($api_key)) {
            $params = $request->get_json_params();
            $api_key = $params['api_key'] ?? '';
        }
        
        if (empty($api_key)) {
            return false;
        }
        
        // Verify API key exists in database
        global $wpdb;
        $table = $wpdb->prefix . 'sl_boxes';
        $box = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM $table WHERE api_key = %s AND is_active = 1",
            $api_key
        ));
        
        return $box !== null;
    }
    
    /**
     * Validate request security (FCrDNS and timestamp)
     * 
     * @param WP_REST_Request $request The incoming request
     * @return true|WP_Error True if valid, WP_Error if security check fails
     */
    public static function validate_request_security($request) {
        // Check timestamp first (always required when present)
        $timestamp_result = self::validate_timestamp($request);
        if (is_wp_error($timestamp_result)) {
            return $timestamp_result;
        }
        
        // Check DNS verification if enabled
        if (get_option('sl_payments_verify_sl_origin', true)) {
            $dns_result = self::validate_sl_origin();
            if (is_wp_error($dns_result)) {
                return $dns_result;
            }
        }
        
        return true;
    }
    
    /**
     * Validate request timestamp to prevent replay attacks
     * 
     * @param WP_REST_Request $request The incoming request
     * @return true|WP_Error True if valid, WP_Error if timestamp is stale
     */
    public static function validate_timestamp($request) {
        // Get timestamp from request body or query params
        $params = $request->get_json_params();
        $timestamp = $params['timestamp'] ?? $request->get_param('timestamp');
        
        // If no timestamp provided, allow for backwards compatibility
        // but log a warning in debug mode
        if (empty($timestamp)) {
            if (get_option('sl_payments_debug_mode', false)) {
                error_log('SL Payments: Request missing timestamp (backwards compatibility mode)');
            }
            return true;
        }
        
        $request_time = intval($timestamp);
        $server_time = time();
        $difference = abs($server_time - $request_time);
        $tolerance = intval(get_option('sl_payments_timestamp_tolerance', self::TIMESTAMP_TOLERANCE));
        
        if ($difference > $tolerance) {
            return new WP_Error(
                'timestamp_expired',
                sprintf(
                    'Request timestamp too old or too far in future. Difference: %d seconds (max: %d)',
                    $difference,
                    $tolerance
                ),
                array('status' => 403)
            );
        }
        
        return true;
    }
    
    /**
     * Validate that request originates from Second Life servers using FCrDNS
     * (Forward-Confirmed Reverse DNS)
     * 
     * @return true|WP_Error True if valid, WP_Error if origin cannot be verified
     */
    public static function validate_sl_origin() {
        // Get the remote IP address
        $remote_ip = self::get_remote_ip();
        
        if (empty($remote_ip)) {
            return new WP_Error(
                'no_remote_ip',
                'Could not determine remote IP address',
                array('status' => 403)
            );
        }
        
        // Allow requests from localhost / same server (for internal WordPress requests)
        if (self::is_local_request($remote_ip)) {
            if (get_option('sl_payments_debug_mode', false)) {
                error_log("SL Payments: Allowing local request from {$remote_ip}");
            }
            return true;
        }
        
        // Step 1: Reverse DNS lookup (IP -> hostname)
        $hostname = gethostbyaddr($remote_ip);
        
        // If gethostbyaddr returns the IP unchanged, reverse DNS failed
        if ($hostname === $remote_ip) {
            if (get_option('sl_payments_debug_mode', false)) {
                error_log("SL Payments: Reverse DNS failed for IP: {$remote_ip}");
            }
            return new WP_Error(
                'reverse_dns_failed',
                'Reverse DNS lookup failed for remote IP',
                array('status' => 403, 'ip' => $remote_ip)
            );
        }
        
        // Step 2: Check if hostname matches Second Life patterns
        $hostname_valid = false;
        foreach (self::$valid_sl_hostname_patterns as $pattern) {
            if (preg_match($pattern, $hostname)) {
                $hostname_valid = true;
                break;
            }
        }
        
        if (!$hostname_valid) {
            if (get_option('sl_payments_debug_mode', false)) {
                error_log("SL Payments: Hostname '{$hostname}' does not match SL patterns");
            }
            return new WP_Error(
                'invalid_hostname',
                'Request does not originate from Second Life servers',
                array('status' => 403, 'hostname' => $hostname)
            );
        }
        
        // Step 3: Forward DNS lookup (hostname -> IP) to confirm
        $forward_ips = gethostbynamel($hostname);
        
        if ($forward_ips === false || !in_array($remote_ip, $forward_ips)) {
            if (get_option('sl_payments_debug_mode', false)) {
                error_log("SL Payments: FCrDNS failed - IP {$remote_ip} not in forward lookup for {$hostname}");
            }
            return new WP_Error(
                'fcrdns_failed',
                'Forward-confirmed reverse DNS validation failed',
                array('status' => 403, 'ip' => $remote_ip, 'hostname' => $hostname)
            );
        }
        
        if (get_option('sl_payments_debug_mode', false)) {
            error_log("SL Payments: FCrDNS passed - {$remote_ip} verified as {$hostname}");
        }
        
        return true;
    }
    
    /**
     * Get the remote IP address, handling proxies
     * 
     * @return string|null The remote IP address or null if not determinable
     */
    private static function get_remote_ip() {
        // Check for proxy headers (in order of trust)
        // Note: These can be spoofed, but our FCrDNS check validates the actual connection
        $headers = array(
            'HTTP_CF_CONNECTING_IP',     // Cloudflare
            'HTTP_X_REAL_IP',            // Nginx proxy
            'HTTP_X_FORWARDED_FOR',      // Standard proxy header
            'REMOTE_ADDR',               // Direct connection
        );
        
        foreach ($headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ip = $_SERVER[$header];
                
                // X-Forwarded-For can contain multiple IPs, take the first (client)
                if ($header === 'HTTP_X_FORWARDED_FOR') {
                    $ips = explode(',', $ip);
                    $ip = trim($ips[0]);
                }
                
                // Validate it looks like an IP
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        
        return null;
    }
    
    /**
     * Check if the request is from localhost or the same server
     * 
     * @param string $ip The remote IP address
     * @return bool True if the request is local
     */
    private static function is_local_request($ip) {
        // Check for localhost IPs
        $local_ips = array(
            '127.0.0.1',
            '::1',
        );
        
        if (in_array($ip, $local_ips)) {
            return true;
        }
        
        // Check if it's a private/internal IP
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) === false) {
            // This means it IS a private or reserved range IP
            // Could be same-server requests on internal network
            
            // Get the server's own IP to compare
            $server_ip = $_SERVER['SERVER_ADDR'] ?? '';
            if ($ip === $server_ip) {
                return true;
            }
            
            // Allow loopback and link-local addresses
            if (strpos($ip, '127.') === 0 || strpos($ip, '10.') === 0 || 
                strpos($ip, '192.168.') === 0 || strpos($ip, '172.') === 0) {
                return true;
            }
        }
        
        // Check if the request IP matches the server's public IP (same-origin request)
        $server_ip = $_SERVER['SERVER_ADDR'] ?? '';
        if (!empty($server_ip) && $ip === $server_ip) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Register a payment box
     */
    public static function register_box($request) {
        global $wpdb;
        $table = $wpdb->prefix . 'sl_boxes';
        
        $params = $request->get_json_params();
        $api_key = $params['api_key'] ?? $request->get_header('X-SL-API-Key');
        
        if (empty($api_key)) {
            return new WP_Error('missing_api_key', 'API key is required', array('status' => 400));
        }
        
        // Position uses angle brackets which sanitize_text_field strips
        // So we sanitize it manually - only allow numbers, commas, spaces, periods, and angle brackets
        $raw_position = $params['position'] ?? '';
        $clean_position = preg_replace('/[^0-9.,<>\s-]/', '', $raw_position);
        
        $data = array(
            'url' => sanitize_url($params['url'] ?? ''),
            'owner_key' => sanitize_text_field($params['owner_key'] ?? ''),
            'owner_name' => sanitize_text_field($params['owner_name'] ?? ''),
            'object_name' => sanitize_text_field($params['object_name'] ?? ''),
            'region' => sanitize_text_field($params['region'] ?? ''),
            'position' => $clean_position,
            'last_seen' => current_time('mysql', true), // GMT for consistency
        );
        
        // Check if box exists
        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM $table WHERE api_key = %s",
            $api_key
        ));
        
        if ($existing) {
            // Update existing box
            $wpdb->update($table, $data, array('api_key' => $api_key));
            $box_id = $existing->id;
        } else {
            // Insert new box
            $data['api_key'] = $api_key;
            $wpdb->insert($table, $data);
            $box_id = $wpdb->insert_id;
        }
        
        return rest_ensure_response(array(
            'success' => true,
            'box_id' => $box_id,
            'message' => 'Box registered successfully',
        ));
    }
    
    /**
     * Heartbeat to keep box URL fresh
     */
    public static function heartbeat($request) {
        global $wpdb;
        $table = $wpdb->prefix . 'sl_boxes';
        
        $params = $request->get_json_params();
        $api_key = $params['api_key'] ?? $request->get_header('X-SL-API-Key');
        
        $data = array(
            'last_seen' => current_time('mysql', true), // GMT for consistency
        );
        
        // Update URL if provided
        if (!empty($params['url'])) {
            $data['url'] = sanitize_url($params['url']);
        }
        
        $wpdb->update($table, $data, array('api_key' => $api_key));
        
        return rest_ensure_response(array(
            'success' => true,
            'message' => 'Heartbeat received',
        ));
    }
    
    /**
     * Get pending payments for a box
     */
    public static function get_pending($request) {
        // Validate security (this endpoint has __return_true permission callback)
        $security_check = self::validate_request_security($request);
        if (is_wp_error($security_check)) {
            if (get_option('sl_payments_debug_mode', false)) {
                error_log('SL Payments Security (get_pending): ' . $security_check->get_error_message());
            }
            return $security_check;
        }
        
        global $wpdb;
        $table_txn = $wpdb->prefix . 'sl_transactions';
        $table_boxes = $wpdb->prefix . 'sl_boxes';
        
        $api_key = $request->get_param('api_key');
        
        // Verify API key and get box
        $box = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM $table_boxes WHERE api_key = %s AND is_active = 1",
            $api_key
        ));
        
        if (!$box) {
            if (get_option('sl_payments_debug_mode', false)) {
                error_log("SL Payments: get_pending - Invalid API key: " . $api_key);
            }
            return new WP_Error('invalid_api_key', 'Invalid API key', array('status' => 401));
        }
        
        // Update last seen and position if provided
        $update_data = array('last_seen' => current_time('mysql', true)); // Use GMT for consistency
        
        // Check for position in query params (sent by box during polling)
        $position = $request->get_param('position');
        $region = $request->get_param('region');
        
        if (!empty($position)) {
            // URL decode and sanitize (preserve angle brackets)
            $position = urldecode($position);
            $clean_position = preg_replace('/[^0-9.,<>\s-]/', '', $position);
            $update_data['position'] = $clean_position;
        }
        if (!empty($region)) {
            $region = urldecode($region);
            $update_data['region'] = sanitize_text_field($region);
        }
        
        $wpdb->update($table_boxes, $update_data, array('id' => $box->id));
        
        // Get oldest pending transaction for this box that hasn't expired
        $now = gmdate('Y-m-d H:i:s'); // Use PHP's gmdate for consistency
        
        // First check if there are ANY pending transactions for this box (for debugging)
        $any_pending = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table_txn WHERE box_id = %d AND status = 'pending'",
            $box->id
        ));
        
        // Now get the actual transaction with expiry check
        // Order by created_at DESC to prioritize newest transactions (avoids old stuck ones)
        $transaction = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_txn 
             WHERE box_id = %d AND status = 'pending' 
             AND (expires_at IS NULL OR expires_at > %s)
             ORDER BY created_at DESC LIMIT 1",
            $box->id, $now
        ));
        
        // Debug: If there are pending but none returned, they're likely expired
        if ($any_pending > 0 && !$transaction) {
            // Get the most recent one to see why it's not matching
            $debug_txn = $wpdb->get_row($wpdb->prepare(
                "SELECT transaction_id, expires_at, status FROM $table_txn 
                 WHERE box_id = %d AND status = 'pending' 
                 ORDER BY created_at DESC LIMIT 1",
                $box->id
            ));
            if (get_option('sl_payments_debug_mode', false)) {
                error_log("SL Payments DEBUG: {$any_pending} pending transactions for box {$box->id}, but none valid.");
                error_log("SL Payments DEBUG: Server time: {$now}");
                error_log("SL Payments DEBUG: Latest txn expires_at: " . ($debug_txn ? $debug_txn->expires_at : 'N/A'));
            }
        }
        
        if (get_option('sl_payments_debug_mode', false)) {
            error_log("SL Payments: get_pending - Box ID: {$box->id}, API Key: {$api_key}");
            error_log("SL Payments: get_pending - Found transaction: " . ($transaction ? $transaction->transaction_id : 'none'));
        }
        
        // Build response
        $response_data = array(
            'has_pending' => false,
            'box_id' => $box->id,
            'server_time' => $now,
            'cache_bust' => time(),
        );
        
        if ($transaction) {
            $response_data = array(
                'has_pending' => true,
                'transaction_id' => $transaction->transaction_id,
                'payer_name' => $transaction->buyer_sl_name,
                'amount' => intval($transaction->amount),
                'timeout' => intval(get_option('sl_payments_payment_timeout', 900)),
                'product_data' => json_decode($transaction->product_data, true),
                'box_id' => $box->id,
                'server_time' => $now,
            );
        }
        
        // Add no-cache headers to prevent hosting caching
        $response = rest_ensure_response($response_data);
        $response->header('Cache-Control', 'no-store, no-cache, must-revalidate, max-age=0');
        $response->header('Pragma', 'no-cache');
        $response->header('Expires', '0');
        
        return $response;
    }
    
    /**
     * Complete a payment
     */
    public static function complete_payment($request) {
        global $wpdb;
        $table_txn = $wpdb->prefix . 'sl_transactions';
        $table_boxes = $wpdb->prefix . 'sl_boxes';
        
        $params = $request->get_json_params();
        $transaction_id = sanitize_text_field($params['transaction_id'] ?? '');
        $api_key = $params['api_key'] ?? $request->get_header('X-SL-API-Key');
        
        if (get_option('sl_payments_debug_mode', false)) {
            error_log("SL Payments: complete_payment called for transaction: " . $transaction_id);
        }
        
        if (empty($transaction_id)) {
            return new WP_Error('missing_transaction_id', 'Transaction ID is required', array('status' => 400));
        }
        
        // Get the box making this request for verification
        $requesting_box = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM $table_boxes WHERE api_key = %s AND is_active = 1",
            $api_key
        ));
        
        if (!$requesting_box) {
            return new WP_Error('invalid_box', 'Invalid or inactive payment box', array('status' => 401));
        }
        
        // Start transaction for atomic operation (prevents race conditions)
        $wpdb->query('START TRANSACTION');
        
        try {
            // Lock the row with FOR UPDATE to prevent concurrent modifications
            $transaction = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM $table_txn WHERE transaction_id = %s FOR UPDATE",
                $transaction_id
            ));
            
            if (!$transaction) {
                $wpdb->query('ROLLBACK');
                if (get_option('sl_payments_debug_mode', false)) {
                    error_log("SL Payments: Transaction not found: " . $transaction_id);
                }
                return new WP_Error('transaction_not_found', 'Transaction not found', array('status' => 404));
            }
            
            // Verify this box owns this transaction
            if ($transaction->box_id !== $requesting_box->id) {
                $wpdb->query('ROLLBACK');
                if (get_option('sl_payments_debug_mode', false)) {
                    error_log("SL Payments: Box {$requesting_box->id} tried to complete transaction belonging to box {$transaction->box_id}");
                }
                return new WP_Error(
                    'wrong_box', 
                    'This transaction belongs to a different payment box', 
                    array('status' => 403)
                );
            }
            
            if (get_option('sl_payments_debug_mode', false)) {
                error_log("SL Payments: Transaction status for {$transaction_id}: {$transaction->status}");
            }
            
            if ($transaction->status !== 'pending') {
                $wpdb->query('ROLLBACK');
                if (get_option('sl_payments_debug_mode', false)) {
                    error_log("SL Payments: Transaction {$transaction_id} not pending - status is: {$transaction->status}");
                }
                return new WP_Error(
                    'invalid_status', 
                    'Transaction is not pending (current status: ' . $transaction->status . ')', 
                    array('status' => 400, 'current_status' => $transaction->status)
                );
            }
            
            // Update transaction (using GMT for consistency)
            $wpdb->update(
                $table_txn,
                array(
                    'status' => 'completed',
                    'buyer_sl_key' => sanitize_text_field($params['payer_key'] ?? ''),
                    'fee' => intval($params['fee'] ?? 0),
                    'completed_at' => gmdate('Y-m-d H:i:s'),
                ),
                array('transaction_id' => $transaction_id)
            );
            
            // Commit the transaction
            $wpdb->query('COMMIT');
            
        } catch (Exception $e) {
            $wpdb->query('ROLLBACK');
            if (get_option('sl_payments_debug_mode', false)) {
                error_log("SL Payments: Exception during complete_payment: " . $e->getMessage());
            }
            return new WP_Error('database_error', 'Database error during payment completion', array('status' => 500));
        }
        
        // Update WooCommerce order if linked (outside transaction - WC has its own)
        if ($transaction->order_id) {
            $order = wc_get_order($transaction->order_id);
            if ($order) {
                $order->payment_complete($transaction_id);
                $order->add_order_note(sprintf(
                    __('Second Life payment received. L$%d from %s (Fee: L$%d)', 'secondlife-payments'),
                    intval($params['amount'] ?? $transaction->amount),
                    sanitize_text_field($params['payer_name'] ?? $transaction->buyer_sl_name),
                    intval($params['fee'] ?? 0)
                ));
            }
        }
        
        // Fire action for other integrations
        do_action('sl_payments_completed', $transaction_id, $params);
        
        return rest_ensure_response(array(
            'success' => true,
            'message' => 'Payment completed',
        ));
    }
    
    /**
     * Match an incoming payment to a pending transaction
     * 
     * This is called by the LSL script when someone pays the box.
     * Instead of pre-locking to one payer, we check ALL pending transactions
     * to find one matching the payer's name and amount.
     * 
     * This allows multiple customers to have pending orders simultaneously
     * without blocking each other.
     */
    public static function match_payment($request) {
        global $wpdb;
        $table_txn = $wpdb->prefix . 'sl_transactions';
        $table_boxes = $wpdb->prefix . 'sl_boxes';
        
        $params = $request->get_json_params();
        $payer_name = sanitize_text_field($params['payer_name'] ?? '');
        $amount = intval($params['amount'] ?? 0);
        $api_key = $params['api_key'] ?? $request->get_header('X-SL-API-Key');
        
        if (get_option('sl_payments_debug_mode', false)) {
            error_log("SL Payments: match_payment called - payer: {$payer_name}, amount: L\${$amount}");
        }
        
        if (empty($payer_name)) {
            return new WP_Error('missing_payer_name', 'Payer name is required', array('status' => 400));
        }
        
        if ($amount <= 0) {
            return new WP_Error('invalid_amount', 'Amount must be positive', array('status' => 400));
        }
        
        // Get the box making this request
        $box = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM $table_boxes WHERE api_key = %s AND is_active = 1",
            $api_key
        ));
        
        if (!$box) {
            return new WP_Error('invalid_box', 'Invalid or inactive payment box', array('status' => 401));
        }
        
        $now = gmdate('Y-m-d H:i:s');
        
        // Start transaction for atomic operation
        $wpdb->query('START TRANSACTION');
        
        try {
            // Find a matching pending transaction for this box
            // Match by: buyer name (case insensitive), amount, not expired, status pending
            // Use FIFO ordering (oldest first) for fairness
            // Lock the row to prevent race conditions with concurrent requests
            $transaction = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM $table_txn 
                 WHERE box_id = %d 
                 AND LOWER(buyer_sl_name) = LOWER(%s)
                 AND amount = %d
                 AND status = 'pending'
                 AND (expires_at IS NULL OR expires_at > %s)
                 ORDER BY created_at ASC 
                 LIMIT 1
                 FOR UPDATE",
                $box->id, $payer_name, $amount, $now
            ));
            
            if (!$transaction) {
                $wpdb->query('ROLLBACK');
                
                if (get_option('sl_payments_debug_mode', false)) {
                    // Debug: check what transactions exist for this payer
                    $debug_txns = $wpdb->get_results($wpdb->prepare(
                        "SELECT transaction_id, amount, status, expires_at FROM $table_txn 
                         WHERE box_id = %d AND LOWER(buyer_sl_name) = LOWER(%s)
                         ORDER BY created_at DESC LIMIT 5",
                        $box->id, $payer_name
                    ));
                    error_log("SL Payments: No match for {$payer_name} L\${$amount}. Existing transactions: " . print_r($debug_txns, true));
                }
                
                return new WP_Error(
                    'no_matching_transaction', 
                    'No pending transaction found for this payer and amount',
                    array('status' => 404, 'payer_name' => $payer_name, 'amount' => $amount)
                );
            }
            
            if (get_option('sl_payments_debug_mode', false)) {
                error_log("SL Payments: Found matching transaction: {$transaction->transaction_id}");
            }
            
            $wpdb->query('COMMIT');
            
        } catch (Exception $e) {
            $wpdb->query('ROLLBACK');
            if (get_option('sl_payments_debug_mode', false)) {
                error_log("SL Payments: Exception during match_payment: " . $e->getMessage());
            }
            return new WP_Error('database_error', 'Database error during payment matching', array('status' => 500));
        }
        
        // Return the matched transaction details
        // The LSL script will then call /complete to finalize
        return rest_ensure_response(array(
            'success' => true,
            'matched' => true,
            'transaction_id' => $transaction->transaction_id,
            'order_id' => $transaction->order_id,
            'expected_amount' => intval($transaction->amount),
            'buyer_name' => $transaction->buyer_sl_name,
        ));
    }
    
    /**
     * Get transaction status (for frontend polling)
     */
    public static function get_status($request) {
        global $wpdb;
        $table_txn = $wpdb->prefix . 'sl_transactions';
        
        $transaction_id = $request->get_param('transaction_id');
        
        $transaction = $wpdb->get_row($wpdb->prepare(
            "SELECT status, amount, buyer_sl_name, created_at, completed_at, expires_at FROM $table_txn WHERE transaction_id = %s",
            $transaction_id
        ));
        
        if (!$transaction) {
            return new WP_Error('transaction_not_found', 'Transaction not found', array('status' => 404));
        }
        
        // Check if expired (expires_at is stored in GMT)
        if ($transaction->status === 'pending' && $transaction->expires_at) {
            // Ensure we compare GMT to GMT
            $expires = strtotime($transaction->expires_at . ' UTC');
            $now = time();
            
            error_log("SL Payments: Status check - expires_at: {$transaction->expires_at} UTC ({$expires}), now: {$now}, diff: " . ($expires - $now) . "s");
            
            if ($expires < $now) {
                // Mark as expired
                $wpdb->update(
                    $table_txn,
                    array('status' => 'expired'),
                    array('transaction_id' => $transaction_id)
                );
                $transaction->status = 'expired';
                error_log("SL Payments: Transaction {$transaction_id} marked as expired");
            }
        }
        
        return rest_ensure_response(array(
            'status' => $transaction->status,
            'amount' => intval($transaction->amount),
            'buyer_name' => $transaction->buyer_sl_name,
            'created_at' => $transaction->created_at,
            'completed_at' => $transaction->completed_at,
        ));
    }
    
    /**
     * Initiate a new payment
     */
    public static function initiate_payment($request) {
        global $wpdb;
        $table_txn = $wpdb->prefix . 'sl_transactions';
        $table_boxes = $wpdb->prefix . 'sl_boxes';
        
        $params = $request->get_json_params();
        
        $buyer_name = sanitize_text_field($params['buyer_name'] ?? '');
        $amount = intval($params['amount'] ?? 0);
        $order_id = intval($params['order_id'] ?? 0);
        $box_api_key = sanitize_text_field($params['box_api_key'] ?? get_option('sl_payments_default_box_key', ''));
        
        if (empty($buyer_name) || $amount <= 0) {
            return new WP_Error('invalid_params', 'Buyer name and amount are required', array('status' => 400));
        }
        
        // Find the box
        $box = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_boxes WHERE api_key = %s AND is_active = 1",
            $box_api_key
        ));
        
        if (!$box) {
            return new WP_Error('no_box', 'No active payment box configured', array('status' => 400));
        }
        
        // Generate transaction ID
        require_once SL_PAYMENTS_PATH . 'includes/class-sl-activator.php';
        $transaction_id = SL_Activator::generate_transaction_id();
        
        // Calculate expiration (use GMT for consistency across timezones)
        $timeout = intval(get_option('sl_payments_payment_timeout', 900));
        $expires_at = gmdate('Y-m-d H:i:s', time() + $timeout);
        
        // Create transaction
        $wpdb->insert($table_txn, array(
            'transaction_id' => $transaction_id,
            'box_id' => $box->id,
            'order_id' => $order_id ?: null,
            'buyer_sl_name' => $buyer_name,
            'amount' => $amount,
            'status' => 'pending',
            'product_data' => json_encode($params['product_data'] ?? array()),
            'expires_at' => $expires_at,
        ));
        
        // Try to notify the box directly
        $box_notified = false;
        if (!empty($box->url)) {
            $response = wp_remote_post($box->url, array(
                'timeout' => 10,
                'headers' => array(
                    'Content-Type' => 'application/json',
                ),
                'body' => json_encode(array(
                    'action' => 'await_payment',
                    'transaction_id' => $transaction_id,
                    'payer_name' => $buyer_name,
                    'amount' => $amount,
                    'timeout' => $timeout,
                )),
            ));
            
            $box_notified = !is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200;
        }
        
        return rest_ensure_response(array(
            'success' => true,
            'transaction_id' => $transaction_id,
            'box_notified' => $box_notified,
            'expires_at' => $expires_at,
            'box_location' => array(
                'region' => $box->region,
                'position' => $box->position,
                'slurl' => self::build_slurl($box->region, $box->position),
            ),
        ));
    }
    
    /**
     * Build SLURL from region and position
     * Uses maps.secondlife.com format for browser compatibility
     */
    private static function build_slurl($region, $position) {
        if (empty($region)) {
            return '';
        }
        
        // Handle region names with spaces
        $region_encoded = rawurlencode($region);
        
        // Default to center of region if no position
        $x = 128;
        $y = 128;
        $z = 25;
        
        if (!empty($position)) {
            // Match formats: <128, 45, 23> or <128.5, 45.2, 23.8>
            if (preg_match('/<\s*([\d.]+)\s*,\s*([\d.]+)\s*,\s*([\d.]+)\s*>/', $position, $matches)) {
                $x = round(floatval($matches[1]));
                $y = round(floatval($matches[2]));
                $z = round(floatval($matches[3]));
            }
        }
        
        // Use maps.secondlife.com for browser compatibility
        return "https://maps.secondlife.com/secondlife/$region_encoded/$x/$y/$z";
    }
}
