<?php
/**
 * WooCommerce Blocks Integration
 * 
 * Adds support for the block-based checkout in WooCommerce
 */

use Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType;

if (!defined('WPINC')) {
    die;
}

/**
 * DBG Payments Blocks Integration
 */
final class SL_WC_Blocks_Support extends AbstractPaymentMethodType {
    
    /**
     * Payment method name/id/slug
     *
     * @var string
     */
    protected $name = 'secondlife';
    
    /**
     * Gateway instance
     *
     * @var SL_WC_Gateway
     */
    private $gateway;
    
    /**
     * Initializes the payment method
     */
    public function initialize() {
        $this->settings = get_option('woocommerce_secondlife_settings', array());
        
        // Load gateway class if not already loaded
        if (!class_exists('SL_WC_Gateway')) {
            require_once SL_PAYMENTS_PATH . 'includes/class-sl-wc-gateway.php';
        }
        
        $gateways = WC()->payment_gateways->payment_gateways();
        $this->gateway = isset($gateways['secondlife']) ? $gateways['secondlife'] : null;
    }
    
    /**
     * Returns if this payment method should be active
     *
     * @return boolean
     */
    public function is_active() {
        if (!$this->gateway) {
            return false;
        }
        return $this->gateway->is_available();
    }
    
    /**
     * Returns an array of scripts/handles to be registered for this payment method
     *
     * @return array
     */
    public function get_payment_method_script_handles() {
        $asset_path = SL_PAYMENTS_PATH . 'assets/js/sl-blocks-checkout.asset.php';
        
        // Force cache bust with file modification time
        $js_file = SL_PAYMENTS_PATH . 'assets/js/sl-blocks-checkout.js';
        $version = file_exists($js_file) ? filemtime($js_file) : SL_PAYMENTS_VERSION;
        $dependencies = array('wc-blocks-registry', 'wc-settings', 'wp-element', 'wp-html-entities', 'wp-i18n');
        
        // Check if asset file exists (generated by build process)
        if (file_exists($asset_path)) {
            $asset = require $asset_path;
            $version = isset($asset['version']) ? $asset['version'] : $version;
            $dependencies = isset($asset['dependencies']) ? $asset['dependencies'] : $dependencies;
        }
        
        wp_register_script(
            'sl-payments-blocks',
            SL_PAYMENTS_URL . 'assets/js/sl-blocks-checkout.js',
            $dependencies,
            $version,
            true
        );
        
        // Set script translations if available
        if (function_exists('wp_set_script_translations')) {
            wp_set_script_translations('sl-payments-blocks', 'secondlife-payments');
        }
        
        return array('sl-payments-blocks');
    }
    
    /**
     * Returns an array of key=>value pairs of data made available to the payment methods script
     *
     * @return array
     */
    public function get_payment_method_data() {
        // Get L$/USD exchange rate
        $linden_rate = 250;
        if (class_exists('SL_Admin')) {
            $linden_rate = SL_Admin::get_exchange_rate();
        } else {
            $linden_rate = floatval(get_option('sl_payments_exchange_rate', 250));
        }
        
        // Get store currency
        $store_currency = get_woocommerce_currency();
        
        // Calculate currency to USD conversion factor
        // If store currency is USD, factor is 1.0
        // Otherwise, we need to convert: factor = 1 / rate[currency]
        $currency_to_usd_factor = 1.0;
        
        if ($store_currency !== 'USD' && class_exists('SL_Admin')) {
            $usd_amount = SL_Admin::convert_to_usd(1.0, $store_currency);
            if ($usd_amount !== false) {
                $currency_to_usd_factor = $usd_amount;
            }
        }
        
        return array(
            'title' => $this->get_setting('title', __('Pay with Linden Dollars (L$)', 'secondlife-payments')),
            'description' => $this->get_setting('description', __('Pay using Linden Dollars in Second Life.', 'secondlife-payments')),
            'supports' => array_filter($this->gateway ? $this->gateway->supports : array(), array($this->gateway, 'supports')),
            'lindenRate' => $linden_rate,              // L$ per $1 USD
            'currencyToUsdFactor' => $currency_to_usd_factor, // Store currency to USD conversion
            'currency' => $store_currency,
            // Legacy: keep exchangeRate for backwards compatibility
            'exchangeRate' => $linden_rate,
        );
    }
}
