<?php
/**
 * WooCommerce Payment Gateway for Second Life Linden Dollars
 */

if (!defined('WPINC')) {
    die;
}

// Make sure WooCommerce is active
if (!class_exists('WC_Payment_Gateway')) {
    return;
}

class SL_WC_Gateway extends WC_Payment_Gateway {
    
    /**
     * Exchange rate (L$ per $1 USD)
     * @var float
     */
    protected $exchange_rate;
    
    /**
     * API key for the payment box
     * @var string
     */
    protected $box_api_key;
    
    /**
     * Payment timeout in seconds
     * @var int
     */
    protected $payment_timeout;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->id = 'secondlife';
        $this->icon = ''; // Optional icon URL
        $this->has_fields = true;
        $this->method_title = __('Second Life (Linden Dollars)', 'secondlife-payments');
        $this->method_description = __('Accept payments in Linden Dollars (L$) from Second Life users.', 'secondlife-payments');
        
        // Supported features
        $this->supports = array(
            'products',
        );
        
        // Load settings
        $this->init_form_fields();
        $this->init_settings();
        
        // Get settings - gateway only has display settings, everything else from SL Payments
        $this->title = $this->get_option('title');
        $this->description = $this->get_option('description');
        $this->enabled = $this->get_option('enabled');
        
        // All config comes from SL Payments settings
        $this->box_api_key = get_option('sl_payments_default_box_key', '');
        $this->payment_timeout = intval(get_option('sl_payments_payment_timeout', 900));
        $this->exchange_rate = $this->get_current_exchange_rate();
        
        // Hooks
        add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
        add_action('woocommerce_thankyou_' . $this->id, array($this, 'thankyou_page'));
        add_action('woocommerce_receipt_' . $this->id, array($this, 'receipt_page'));
        
        // Add endpoint for payment pending page
        add_action('init', array($this, 'add_endpoints'));
        add_action('template_redirect', array($this, 'handle_payment_pending'));
    }
    
    /**
     * Check if the gateway is available for use
     * 
     * @return bool
     */
    public function is_available() {
        // First check parent availability (enabled, etc.)
        if (!parent::is_available()) {
            return false;
        }
        
        // Check if WooCommerce is available
        if (!function_exists('WC')) {
            return false;
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'sl_boxes';
        
        // Check if round-robin mode is enabled
        $round_robin = get_option('sl_payments_round_robin', false);
        
        if ($round_robin) {
            // Round-robin mode: just need at least one active box
            $active_count = $wpdb->get_var(
                "SELECT COUNT(*) FROM $table WHERE is_active = 1"
            );
            
            if ($active_count == 0) {
                if (get_option('sl_payments_debug_mode', false)) {
                    error_log('SL Payments: Gateway unavailable - no active payment boxes for round-robin');
                }
                return false;
            }
            
            return true;
        }
        
        // Single box mode: check the configured default box
        if (empty($this->box_api_key)) {
            if (get_option('sl_payments_debug_mode', false)) {
                error_log('SL Payments: Gateway unavailable - no payment box configured');
            }
            return false;
        }
        
        $box = $wpdb->get_row($wpdb->prepare(
            "SELECT id, is_active FROM $table WHERE api_key = %s",
            $this->box_api_key
        ));
        
        if (!$box) {
            if (get_option('sl_payments_debug_mode', false)) {
                error_log('SL Payments: Gateway unavailable - payment box not found in database');
            }
            return false;
        }
        
        if (!$box->is_active) {
            if (get_option('sl_payments_debug_mode', false)) {
                error_log('SL Payments: Gateway unavailable - payment box is disabled');
            }
            return false;
        }
        
        return true;
    }
    
    /**
     * Admin notice if gateway is enabled but not configured
     */
    public function admin_options() {
        $round_robin = get_option('sl_payments_round_robin', false);
        
        // Show warning if enabled but not properly configured
        if ($this->enabled === 'yes' && empty($this->box_api_key) && !$round_robin) {
            echo '<div class="notice notice-warning"><p>';
            printf(
                __('<strong>DBG Payments:</strong> No payment box selected. Please configure one in <a href="%s">DBG Payments Settings</a> or enable Round-Robin mode with multiple boxes.', 'secondlife-payments'),
                admin_url('admin.php?page=sl-payments')
            );
            echo '</p></div>';
        }
        
        // Show info if round-robin is enabled
        if ($this->enabled === 'yes' && $round_robin) {
            global $wpdb;
            $table = $wpdb->prefix . 'sl_boxes';
            $count = $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE is_active = 1");
            echo '<div class="notice notice-info"><p>';
            printf(
                __('<strong>DBG Payments:</strong> Round-Robin mode is active with %d payment box(es). Orders will be distributed across available boxes.', 'secondlife-payments'),
                $count
            );
            echo '</p></div>';
        }
        
        parent::admin_options();
    }
    
    /**
     * Initialize gateway settings form fields
     * Note: Most settings are in SL Payments > Settings, not here
     */
    public function init_form_fields() {
        $this->form_fields = array(
            'enabled' => array(
                'title' => __('Enable/Disable', 'secondlife-payments'),
                'type' => 'checkbox',
                'label' => __('Enable DBG Payments (Linden Dollars)', 'secondlife-payments'),
                'default' => 'no',
            ),
            'title' => array(
                'title' => __('Title', 'secondlife-payments'),
                'type' => 'text',
                'description' => __('Payment method title shown at checkout.', 'secondlife-payments'),
                'default' => __('Pay with Linden Dollars (L$)', 'secondlife-payments'),
                'desc_tip' => true,
            ),
            'description' => array(
                'title' => __('Description', 'secondlife-payments'),
                'type' => 'textarea',
                'description' => __('Payment method description shown at checkout.', 'secondlife-payments'),
                'default' => __('Pay using Linden Dollars in Second Life. You will need to visit our in-world payment box to complete the payment.', 'secondlife-payments'),
                'desc_tip' => true,
            ),
            'settings_link' => array(
                'title' => __('Configuration', 'secondlife-payments'),
                'type' => 'title',
                'description' => sprintf(
                    __('Configure exchange rate, payment timeout, and payment box in <a href="%s">DBG Payments Settings</a>.', 'secondlife-payments'),
                    admin_url('admin.php?page=sl-payments')
                ),
            ),
        );
    }
    
    /**
     * Payment fields shown at checkout
     */
    public function payment_fields() {
        // Show description
        if ($this->description) {
            echo wpautop(wptexturize($this->description));
        }
        
        // Calculate L$ amount
        $order_total = WC()->cart->get_total('edit');
        $currency = get_woocommerce_currency();
        $linden_amount = $this->convert_to_linden($order_total, $currency);
        
        // Check if conversion failed
        if ($linden_amount === false) {
            ?>
            <div class="sl-payment-error" style="color: #d63638; padding: 10px; background: #fcf0f0; border-left: 4px solid #d63638; margin-bottom: 15px;">
                <strong><?php _e('Currency conversion unavailable', 'secondlife-payments'); ?></strong>
                <p><?php printf(__('Unable to convert %s to Linden Dollars. Please try again later or contact support.', 'secondlife-payments'), $currency); ?></p>
            </div>
            <?php
            return;
        }
        
        ?>
        <div class="sl-payment-fields">
            <p class="sl-linden-amount">
                <strong><?php _e('Amount in Linden Dollars:', 'secondlife-payments'); ?></strong>
                L$<?php echo number_format($linden_amount); ?>
                <?php if ($currency !== 'USD'): ?>
                    <br><small style="color: #666;">
                        <?php printf(__('(Converted from %s %.2f)', 'secondlife-payments'), $currency, $order_total); ?>
                    </small>
                <?php endif; ?>
            </p>
            
            <p class="form-row form-row-first">
                <label for="sl_first_name"><?php _e('Second Life First Name', 'secondlife-payments'); ?> <span class="required">*</span></label>
                <input type="text" class="input-text" name="sl_first_name" id="sl_first_name" 
                       placeholder="<?php esc_attr_e('Your SL first name', 'secondlife-payments'); ?>" required>
            </p>
            
            <p class="form-row form-row-last">
                <label for="sl_last_name"><?php _e('Second Life Last Name', 'secondlife-payments'); ?> <span class="required">*</span></label>
                <input type="text" class="input-text" name="sl_last_name" id="sl_last_name" 
                       placeholder="<?php esc_attr_e('Your SL last name (or Resident)', 'secondlife-payments'); ?>" required>
            </p>
            
            <div class="clear"></div>
            
            <p class="sl-payment-note">
                <small><?php _e('Enter your Second Life avatar name exactly as it appears in-world. After placing your order, you will be directed to pay at our in-world location.', 'secondlife-payments'); ?></small>
            </p>
        </div>
        <?php
    }
    
    /**
     * Validate payment fields
     */
    public function validate_fields() {
        $first_name = sanitize_text_field($_POST['sl_first_name'] ?? '');
        $last_name = sanitize_text_field($_POST['sl_last_name'] ?? '');
        
        if (empty($first_name)) {
            wc_add_notice(__('Please enter your Second Life first name.', 'secondlife-payments'), 'error');
            return false;
        }
        
        if (empty($last_name)) {
            wc_add_notice(__('Please enter your Second Life last name.', 'secondlife-payments'), 'error');
            return false;
        }
        
        // Basic validation - SL names are alphanumeric
        if (!preg_match('/^[a-zA-Z0-9]+$/', $first_name)) {
            wc_add_notice(__('Invalid Second Life first name. Use only letters and numbers.', 'secondlife-payments'), 'error');
            return false;
        }
        
        if (!preg_match('/^[a-zA-Z0-9]+$/', $last_name)) {
            wc_add_notice(__('Invalid Second Life last name. Use only letters and numbers.', 'secondlife-payments'), 'error');
            return false;
        }
        
        return true;
    }
    
    /**
     * Process the payment
     */
    public function process_payment($order_id) {
        $order = wc_get_order($order_id);
        
        $first_name = sanitize_text_field($_POST['sl_first_name'] ?? '');
        $last_name = sanitize_text_field($_POST['sl_last_name'] ?? '');
        $sl_name = $first_name . ' ' . $last_name;
        
        // Calculate L$ amount (use order currency, not store currency, in case they differ)
        $linden_amount = $this->convert_to_linden($order->get_total(), $order->get_currency());
        
        // Save SL name to order meta
        $order->update_meta_data('_sl_buyer_name', $sl_name);
        $order->update_meta_data('_sl_linden_amount', $linden_amount);
        $order->save();
        
        // Create pending transaction via API
        $result = $this->initiate_transaction($order, $sl_name, $linden_amount);
        
        if (is_wp_error($result)) {
            wc_add_notice($result->get_error_message(), 'error');
            return array('result' => 'failure');
        }
        
        // Save transaction ID to order
        $order->update_meta_data('_sl_transaction_id', $result['transaction_id']);
        $order->save();
        
        // Mark order as pending payment
        $order->update_status('pending', __('Awaiting Second Life payment', 'secondlife-payments'));
        
        // Empty cart
        WC()->cart->empty_cart();
        
        // Return to payment pending page
        return array(
            'result' => 'success',
            'redirect' => add_query_arg(array(
                'sl_payment' => $result['transaction_id'],
            ), $order->get_checkout_order_received_url()),
        );
    }
    
    /**
     * Initiate a transaction
     */
    private function initiate_transaction($order, $sl_name, $linden_amount) {
        global $wpdb;
        $table_txn = $wpdb->prefix . 'sl_transactions';
        $table_boxes = $wpdb->prefix . 'sl_boxes';
        
        // Select payment box (with round-robin if enabled)
        $box = $this->select_payment_box();
        
        if (!$box) {
            return new WP_Error('no_box', __('Payment box not configured. Please contact the store owner.', 'secondlife-payments'));
        }
        
        // Generate transaction ID
        require_once SL_PAYMENTS_PATH . 'includes/class-sl-activator.php';
        $transaction_id = SL_Activator::generate_transaction_id();
        
        // Calculate expiration (use GMT for consistency)
        $expires_at = gmdate('Y-m-d H:i:s', time() + $this->payment_timeout);
        
        // Create transaction
        $insert_result = $wpdb->insert($table_txn, array(
            'transaction_id' => $transaction_id,
            'box_id' => $box->id,
            'order_id' => $order->get_id(),
            'buyer_sl_name' => $sl_name,
            'amount' => $linden_amount,
            'status' => 'pending',
            'product_data' => json_encode(array(
                'order_id' => $order->get_id(),
                'order_total' => $order->get_total(),
                'currency' => $order->get_currency(),
            )),
            'expires_at' => $expires_at,
        ));
        
        if (get_option('sl_payments_debug_mode', false)) {
            error_log("SL Payments: Created transaction {$transaction_id} for box_id {$box->id}, amount L\${$linden_amount}, buyer: {$sl_name}");
            error_log("SL Payments: Insert result: " . ($insert_result ? 'success' : 'failed - ' . $wpdb->last_error));
        }
        
        // Try to notify the box directly
        $box_notified = false;
        if (!empty($box->url)) {
            $response = wp_remote_post($box->url, array(
                'timeout' => 10,
                'headers' => array(
                    'Content-Type' => 'application/json',
                ),
                'body' => json_encode(array(
                    'action' => 'await_payment',
                    'transaction_id' => $transaction_id,
                    'payer_name' => $sl_name,
                    'amount' => $linden_amount,
                    'timeout' => $this->payment_timeout,
                )),
            ));
            
            $box_notified = !is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200;
        }
        
        // Build SLURL for the assigned box (not the global override)
        $slurl = '';
        if (!empty($box->region)) {
            $slurl = $this->build_slurl($box->region, $box->position);
        }
        // Fall back to global override only if box has no region
        if (empty($slurl)) {
            $slurl = get_option('sl_payments_box_slurl', '');
        }
        
        return array(
            'transaction_id' => $transaction_id,
            'box_notified' => $box_notified,
            'expires_at' => $expires_at,
            'slurl' => $slurl,
            'region' => $box->region,
            'box_id' => $box->id,
        );
    }
    
    /**
     * Select a payment box using round-robin distribution
     * 
     * When multiple boxes are active and online, distributes orders across them
     * to spread customer traffic across different SL locations.
     * 
     * @return object|null The selected box or null if none available
     */
    private function select_payment_box() {
        global $wpdb;
        $table_boxes = $wpdb->prefix . 'sl_boxes';
        
        // Check if round-robin is enabled
        $round_robin_enabled = get_option('sl_payments_round_robin', false);
        
        if (!$round_robin_enabled) {
            // Use the configured default box
            if (empty($this->box_api_key)) {
                return null;
            }
            return $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM $table_boxes WHERE api_key = %s AND is_active = 1",
                $this->box_api_key
            ));
        }
        
        // Round-robin mode: get all active, online boxes
        // Consider a box "online" if seen in the last 2 minutes
        $online_threshold = gmdate('Y-m-d H:i:s', time() - 120);
        
        $boxes = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table_boxes 
             WHERE is_active = 1 
             AND last_seen IS NOT NULL 
             AND last_seen > %s
             ORDER BY id ASC",
            $online_threshold
        ));
        
        if (empty($boxes)) {
            // No online boxes - fall back to any active box
            $boxes = $wpdb->get_results(
                "SELECT * FROM $table_boxes WHERE is_active = 1 ORDER BY id ASC"
            );
        }
        
        if (empty($boxes)) {
            return null;
        }
        
        // If only one box, use it
        if (count($boxes) === 1) {
            return $boxes[0];
        }
        
        // Multiple boxes - select using round-robin
        $last_box_id = intval(get_option('sl_payments_last_box_id', 0));
        $selected_box = null;
        $found_last = false;
        
        // Find the next box after the last used one
        foreach ($boxes as $box) {
            if ($found_last) {
                $selected_box = $box;
                break;
            }
            if ($box->id == $last_box_id) {
                $found_last = true;
            }
        }
        
        // If we didn't find a next box (last was the final one, or not found), use the first
        if (!$selected_box) {
            $selected_box = $boxes[0];
        }
        
        // Update last used box
        update_option('sl_payments_last_box_id', $selected_box->id, false);
        
        if (get_option('sl_payments_debug_mode', false)) {
            error_log("SL Payments: Round-robin selected box {$selected_box->id} ({$selected_box->region}) from " . count($boxes) . " available boxes");
        }
        
        return $selected_box;
    }
    
    /**
     * Get current exchange rate (live or manual)
     */
    private function get_current_exchange_rate() {
        if (class_exists('SL_Admin')) {
            return SL_Admin::get_exchange_rate();
        }
        return floatval(get_option('sl_payments_exchange_rate', 250));
    }
    
    /**
     * Convert order total to Linden Dollars
     * Supports any currency that WooCommerce is configured for
     * 
     * @param float $amount The order amount
     * @param string|null $currency Optional currency code, defaults to store currency
     * @return int The amount in L$ (rounded up)
     */
    private function convert_to_linden($amount, $currency = null) {
        if ($currency === null) {
            $currency = get_woocommerce_currency();
        }
        
        $debug = get_option('sl_payments_debug_mode', false);
        
        // For USD, no currency conversion needed - just apply L$/USD rate
        if (strtoupper($currency) === 'USD') {
            if ($debug) {
                error_log("SL Payments: Direct USD conversion: {$amount} USD * {$this->exchange_rate} = " . ceil($amount * $this->exchange_rate) . " L$");
            }
            return (int) ceil($amount * $this->exchange_rate);
        }
        
        // For other currencies, first convert to USD, then to L$
        if (class_exists('SL_Admin')) {
            // Try to convert to USD first
            $usd_amount = SL_Admin::convert_to_usd($amount, $currency);
            
            if ($usd_amount !== false) {
                $linden_amount = (int) ceil($usd_amount * $this->exchange_rate);
                if ($debug) {
                    error_log("SL Payments: Currency conversion: {$amount} {$currency} = {$usd_amount} USD = {$linden_amount} L$");
                }
                return $linden_amount;
            }
            
            // Log conversion failure
            if ($debug) {
                error_log("SL Payments: Currency conversion failed for {$currency}, falling back to direct conversion (treating as USD)");
            }
        } else {
            if ($debug) {
                error_log("SL Payments: SL_Admin class not found, falling back to direct conversion");
            }
        }
        
        // Fallback: treat as USD if conversion fails (will be inaccurate for non-USD)
        // This maintains backwards compatibility
        return (int) ceil($amount * $this->exchange_rate);
    }
    
    /**
     * Build SLURL from region and position
     * Uses maps.secondlife.com format for browser compatibility
     */
    private function build_slurl($region, $position) {
        if (empty($region)) {
            return '';
        }
        
        // Handle region names with spaces
        $region_encoded = rawurlencode($region);
        
        // Default to center of region if no position
        $x = 128;
        $y = 128;
        $z = 25;
        
        if (!empty($position)) {
            // Match formats: <128, 45, 23> or <128.5, 45.2, 23.8>
            if (preg_match('/<\s*([\d.]+)\s*,\s*([\d.]+)\s*,\s*([\d.]+)\s*>/', $position, $matches)) {
                $x = round(floatval($matches[1]));
                $y = round(floatval($matches[2]));
                $z = round(floatval($matches[3]));
            }
        }
        
        // Use maps.secondlife.com for browser compatibility
        return "https://maps.secondlife.com/secondlife/$region_encoded/$x/$y/$z";
    }
    
    /**
     * Receipt/Payment pending page
     */
    public function receipt_page($order_id) {
        $order = wc_get_order($order_id);
        $transaction_id = $order->get_meta('_sl_transaction_id');
        
        if (!$transaction_id) {
            echo '<p>' . __('Transaction not found.', 'secondlife-payments') . '</p>';
            return;
        }
        
        $this->render_payment_pending($order, $transaction_id);
    }
    
    /**
     * Thank you page - show payment pending if not completed
     */
    public function thankyou_page($order_id) {
        $order = wc_get_order($order_id);
        
        if ($order->get_status() === 'pending') {
            $transaction_id = isset($_GET['sl_payment']) ? sanitize_text_field($_GET['sl_payment']) : $order->get_meta('_sl_transaction_id');
            
            if ($transaction_id) {
                $this->render_payment_pending($order, $transaction_id);
            }
        }
    }
    
    /**
     * Render payment pending UI
     */
    private function render_payment_pending($order, $transaction_id) {
        global $wpdb;
        $table_txn = $wpdb->prefix . 'sl_transactions';
        $table_boxes = $wpdb->prefix . 'sl_boxes';
        
        $transaction = $wpdb->get_row($wpdb->prepare(
            "SELECT t.*, b.region, b.position FROM $table_txn t 
             LEFT JOIN $table_boxes b ON t.box_id = b.id
             WHERE t.transaction_id = %s",
            $transaction_id
        ));
        
        if (!$transaction) {
            echo '<p>' . __('Transaction not found.', 'secondlife-payments') . '</p>';
            return;
        }
        
        // Build SLURL
        $slurl = get_option('sl_payments_box_slurl', '');
        if (empty($slurl) && !empty($transaction->region)) {
            $slurl = $this->build_slurl($transaction->region, $transaction->position);
        }
        
        ?>
        <div class="sl-payment-pending" data-transaction-id="<?php echo esc_attr($transaction_id); ?>">
            <div class="sl-payment-status">
                <div class="sl-spinner"></div>
                <h3><?php _e('Awaiting Payment in Second Life', 'secondlife-payments'); ?></h3>
            </div>
            
            <div class="sl-payment-details">
                <table>
                    <tr>
                        <th><?php _e('Amount to Pay:', 'secondlife-payments'); ?></th>
                        <td><strong>L$<?php echo number_format($transaction->amount); ?></strong></td>
                    </tr>
                    <tr>
                        <th><?php _e('Pay From Avatar:', 'secondlife-payments'); ?></th>
                        <td><?php echo esc_html($transaction->buyer_sl_name); ?></td>
                    </tr>
                    <?php if ($transaction->region || $slurl): ?>
                    <tr>
                        <th><?php _e('Location:', 'secondlife-payments'); ?></th>
                        <td>
                            <?php if ($transaction->region): ?>
                                <?php echo esc_html($transaction->region); ?>
                                <?php if ($transaction->position): ?>
                                    <br><small><?php echo esc_html($transaction->position); ?></small>
                                <?php endif; ?>
                            <?php elseif ($slurl): ?>
                                <?php _e('See link below', 'secondlife-payments'); ?>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <?php endif; ?>
                </table>
            </div>
            
            <div class="sl-payment-instructions">
                <h4><?php _e('How to Pay:', 'secondlife-payments'); ?></h4>
                <ol>
                    <li><?php _e('Open Second Life and log in as the avatar name shown above', 'secondlife-payments'); ?></li>
                    <?php if ($slurl): ?>
                    <li>
                        <?php _e('Click the button below to teleport to our payment box:', 'secondlife-payments'); ?>
                        <br><br>
                        <a href="<?php echo esc_url($slurl); ?>" class="sl-slurl-button" target="_blank">
                            <?php _e('📍 Teleport to Payment Location', 'secondlife-payments'); ?>
                        </a>
                    </li>
                    <?php else: ?>
                    <li><?php _e('Find our payment box in-world', 'secondlife-payments'); ?></li>
                    <?php endif; ?>
                    <li><?php _e('Right-click the payment box and select "Pay"', 'secondlife-payments'); ?></li>
                    <li><?php printf(__('Enter the exact amount: L$%s', 'secondlife-payments'), number_format($transaction->amount)); ?></li>
                    <li><?php _e('Click "Pay" to complete the transaction', 'secondlife-payments'); ?></li>
                </ol>
                
                <p class="sl-payment-warning">
                    <strong><?php _e('Important:', 'secondlife-payments'); ?></strong>
                    <?php _e('You must pay from the exact avatar name shown above. Payments from other avatars will be refunded.', 'secondlife-payments'); ?>
                </p>
            </div>
            
            <div class="sl-payment-timeout">
                <?php 
                // expires_at is stored in GMT, add UTC suffix for correct parsing
                $expires_timestamp = strtotime($transaction->expires_at . ' UTC');
                $remaining = max(0, $expires_timestamp - time());
                ?>
                <p>
                    <?php _e('Time remaining:', 'secondlife-payments'); ?>
                    <span class="sl-countdown" data-expires="<?php echo esc_attr($expires_timestamp); ?>">
                        <?php echo gmdate('i:s', $remaining); ?>
                    </span>
                </p>
            </div>
        </div>
        
        <style>
        .sl-payment-pending {
            max-width: 600px;
            margin: 20px auto;
            padding: 20px;
            border: 1px solid #ddd;
            border-radius: 8px;
            background: #f9f9f9;
        }
        .sl-payment-status {
            text-align: center;
            margin-bottom: 20px;
        }
        .sl-spinner {
            width: 40px;
            height: 40px;
            border: 4px solid #f3f3f3;
            border-top: 4px solid #3498db;
            border-radius: 50%;
            animation: sl-spin 1s linear infinite;
            margin: 0 auto 15px;
        }
        @keyframes sl-spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        .sl-payment-details table {
            width: 100%;
            margin-bottom: 20px;
        }
        .sl-payment-details th {
            text-align: left;
            padding: 8px;
            width: 40%;
        }
        .sl-payment-details td {
            padding: 8px;
        }
        .sl-payment-instructions {
            background: #fff;
            padding: 15px;
            border-radius: 4px;
            margin-bottom: 15px;
        }
        .sl-payment-instructions ol {
            margin-left: 20px;
        }
        .sl-payment-instructions li {
            margin-bottom: 10px;
        }
        .sl-slurl-button {
            display: inline-block;
            padding: 12px 24px;
            background: #2271b1;
            color: #fff !important;
            text-decoration: none;
            border-radius: 4px;
            font-weight: 600;
            transition: background 0.2s;
        }
        .sl-slurl-button:hover {
            background: #135e96;
            color: #fff !important;
        }
        .sl-payment-warning {
            background: #fff3cd;
            padding: 10px;
            border-left: 4px solid #ffc107;
            margin-top: 15px;
        }
        .sl-payment-timeout {
            text-align: center;
            font-size: 18px;
        }
        .sl-countdown {
            font-weight: bold;
            font-family: monospace;
        }
        .sl-payment-completed .sl-spinner {
            border-top-color: #28a745;
            animation: none;
            border: 4px solid #28a745;
        }
        .sl-payment-completed .sl-spinner::after {
            content: '✓';
            display: block;
            text-align: center;
            line-height: 32px;
            font-size: 24px;
            color: #28a745;
        }
        </style>
        <?php
    }
    
    /**
     * Add custom endpoints
     */
    public function add_endpoints() {
        add_rewrite_endpoint('sl-payment-pending', EP_ROOT | EP_PAGES);
    }
    
    /**
     * Handle payment pending page requests
     */
    public function handle_payment_pending() {
        // This is handled via the thankyou page hook
    }
}
