<?php
/**
 * Plugin Name: DBG Payments
 * Description: Accept Linden Dollar (L$) payments from Second Life users via WooCommerce
 * Version: 0.0.1
 * Author: Drow Bear Goblin Industries
 * License: GPL-2.0+
 * License URI: http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain: secondlife-payments
 * Domain Path: /languages
 * Requires at least: 5.0
 * Requires PHP: 7.4
 * WC requires at least: 5.0
 * WC tested up to: 8.0
 */

// If this file is called directly, abort.
if (!defined('WPINC')) {
    die;
}

// Plugin version - should match the Version header above
define('SL_PAYMENTS_VERSION', '0.0.1');

// Plugin directory path
define('SL_PAYMENTS_PATH', plugin_dir_path(__FILE__));

// Plugin directory URL
define('SL_PAYMENTS_URL', plugin_dir_url(__FILE__));

// Database table names
global $wpdb;
define('SL_PAYMENTS_TABLE_BOXES', $wpdb->prefix . 'sl_boxes');
define('SL_PAYMENTS_TABLE_TRANSACTIONS', $wpdb->prefix . 'sl_transactions');

/**
 * Activation hook
 */
function sl_payments_activate() {
    require_once SL_PAYMENTS_PATH . 'includes/class-sl-activator.php';
    SL_Activator::activate();
}
register_activation_hook(__FILE__, 'sl_payments_activate');

/**
 * Deactivation hook
 */
function sl_payments_deactivate() {
    // Clean up if needed
    flush_rewrite_rules();
}
register_deactivation_hook(__FILE__, 'sl_payments_deactivate');

/**
 * Load plugin textdomain for translations
 */
function sl_payments_load_textdomain() {
    load_plugin_textdomain('secondlife-payments', false, dirname(plugin_basename(__FILE__)) . '/languages');
}
add_action('plugins_loaded', 'sl_payments_load_textdomain');

/**
 * Include required files
 */
function sl_payments_includes() {
    require_once SL_PAYMENTS_PATH . 'includes/class-sl-activator.php';
    require_once SL_PAYMENTS_PATH . 'includes/class-sl-api.php';
    require_once SL_PAYMENTS_PATH . 'includes/class-sl-admin.php';
}
add_action('plugins_loaded', 'sl_payments_includes', 5);

/**
 * Initialize the plugin
 */
function sl_payments_init() {
    // Initialize API endpoints
    SL_API::init();
    
    // Initialize admin if in admin area
    if (is_admin()) {
        SL_Admin::init();
    }
}
add_action('plugins_loaded', 'sl_payments_init', 10);

/**
 * Register WooCommerce payment gateway
 */
function sl_payments_add_gateway($gateways) {
    $gateways[] = 'SL_WC_Gateway';
    return $gateways;
}
add_filter('woocommerce_payment_gateways', 'sl_payments_add_gateway');

/**
 * Load WooCommerce gateway class
 */
function sl_payments_load_gateway() {
    if (class_exists('WC_Payment_Gateway')) {
        require_once SL_PAYMENTS_PATH . 'includes/class-sl-wc-gateway.php';
    }
}
add_action('plugins_loaded', 'sl_payments_load_gateway', 11);

/**
 * Enqueue frontend scripts and styles
 */
function sl_payments_enqueue_scripts() {
    // Only load on checkout and order-pay pages
    if (is_checkout() || is_wc_endpoint_url('order-pay')) {
        wp_enqueue_style(
            'sl-checkout',
            SL_PAYMENTS_URL . 'assets/css/sl-checkout.css',
            array(),
            SL_PAYMENTS_VERSION
        );
        
        wp_enqueue_script(
            'sl-checkout',
            SL_PAYMENTS_URL . 'assets/js/sl-checkout.js',
            array('jquery'),
            SL_PAYMENTS_VERSION,
            true
        );
        
        wp_localize_script('sl-checkout', 'slPayments', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'restUrl' => rest_url('sl-payments/v1/'),
            'nonce' => wp_create_nonce('wp_rest'),
            'i18n' => array(
                'awaiting' => __('Awaiting payment in Second Life...', 'secondlife-payments'),
                'completed' => __('Payment received! Redirecting...', 'secondlife-payments'),
                'timeout' => __('Payment timeout. Please try again.', 'secondlife-payments'),
                'error' => __('An error occurred. Please try again.', 'secondlife-payments'),
            )
        ));
    }
}
add_action('wp_enqueue_scripts', 'sl_payments_enqueue_scripts');

/**
 * Enqueue admin scripts and styles
 */
function sl_payments_enqueue_admin_scripts($hook) {
    // Only load on our admin pages
    if (strpos($hook, 'sl-payments') === false) {
        return;
    }
    
    wp_enqueue_style(
        'sl-admin',
        SL_PAYMENTS_URL . 'assets/css/sl-checkout.css',
        array(),
        SL_PAYMENTS_VERSION
    );
}
add_action('admin_enqueue_scripts', 'sl_payments_enqueue_admin_scripts');

/**
 * Add settings link to plugins page
 */
function sl_payments_plugin_links($links) {
    $settings_link = '<a href="' . admin_url('admin.php?page=sl-payments') . '">' . __('Settings', 'secondlife-payments') . '</a>';
    array_unshift($links, $settings_link);
    return $links;
}
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'sl_payments_plugin_links');

/**
 * Delete SL transaction when WooCommerce order is deleted
 */
function sl_payments_delete_order_transaction($order_id) {
    global $wpdb;
    $table_txn = $wpdb->prefix . 'sl_transactions';
    
    // Delete any transactions associated with this order
    $deleted = $wpdb->delete($table_txn, array('order_id' => $order_id));
    
    if ($deleted) {
        error_log("SL Payments: Deleted {$deleted} transaction(s) for order #{$order_id}");
    }
}
add_action('woocommerce_before_delete_order', 'sl_payments_delete_order_transaction');
add_action('before_delete_post', 'sl_payments_delete_order_transaction'); // For HPOS compatibility
add_action('woocommerce_delete_order', 'sl_payments_delete_order_transaction');

/**
 * Also handle order trashing (move to trash)
 */
function sl_payments_trash_order_transaction($order_id) {
    global $wpdb;
    $table_txn = $wpdb->prefix . 'sl_transactions';
    
    // When order is trashed, mark transaction as cancelled
    $wpdb->update(
        $table_txn, 
        array('status' => 'cancelled'),
        array('order_id' => $order_id, 'status' => 'pending')
    );
}
add_action('woocommerce_trash_order', 'sl_payments_trash_order_transaction');
add_action('wp_trash_post', 'sl_payments_trash_order_transaction');

/**
 * Handle order status change to cancelled
 * This fires when order status is changed to "cancelled" via admin UI or API
 */
function sl_payments_cancel_order_transaction($order_id, $order = null) {
    global $wpdb;
    $table_txn = $wpdb->prefix . 'sl_transactions';
    
    // Mark any pending transaction as cancelled
    $updated = $wpdb->update(
        $table_txn, 
        array('status' => 'cancelled'),
        array('order_id' => $order_id, 'status' => 'pending')
    );
    
    if ($updated && get_option('sl_payments_debug_mode', false)) {
        error_log("SL Payments: Cancelled transaction for order #{$order_id}");
    }
}
add_action('woocommerce_order_status_cancelled', 'sl_payments_cancel_order_transaction', 10, 2);

/**
 * Handle order status change to failed
 */
function sl_payments_fail_order_transaction($order_id, $order = null) {
    global $wpdb;
    $table_txn = $wpdb->prefix . 'sl_transactions';
    
    // Mark any pending transaction as failed
    $wpdb->update(
        $table_txn, 
        array('status' => 'failed'),
        array('order_id' => $order_id, 'status' => 'pending')
    );
}
add_action('woocommerce_order_status_failed', 'sl_payments_fail_order_transaction', 10, 2);

/**
 * Handle any order status change FROM pending
 * This catches edge cases like custom statuses
 */
function sl_payments_order_status_changed($order_id, $old_status, $new_status, $order) {
    // Only process if it was a SL payment order
    if ($order->get_payment_method() !== 'secondlife') {
        return;
    }
    
    // If moving away from pending to anything other than processing/completed, cancel the transaction
    if ($old_status === 'pending' && !in_array($new_status, array('processing', 'completed', 'on-hold'))) {
        global $wpdb;
        $table_txn = $wpdb->prefix . 'sl_transactions';
        
        $wpdb->update(
            $table_txn, 
            array('status' => 'cancelled'),
            array('order_id' => $order_id, 'status' => 'pending')
        );
        
        if (get_option('sl_payments_debug_mode', false)) {
            error_log("SL Payments: Transaction cancelled due to order status change: {$old_status} -> {$new_status}");
        }
    }
}
add_action('woocommerce_order_status_changed', 'sl_payments_order_status_changed', 10, 4);

/**
 * Declare HPOS compatibility for WooCommerce
 */
add_action('before_woocommerce_init', function() {
    if (class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
    }
});

/**
 * Register WooCommerce Blocks integration
 */
add_action('woocommerce_blocks_loaded', function() {
    // Check if the required class exists
    if (!class_exists('Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType')) {
        return;
    }
    
    // Include our blocks integration class
    require_once SL_PAYMENTS_PATH . 'includes/class-sl-wc-blocks.php';
    
    // Register the payment method with WooCommerce Blocks
    add_action(
        'woocommerce_blocks_payment_method_type_registration',
        function(\Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry $payment_method_registry) {
            $payment_method_registry->register(new SL_WC_Blocks_Support());
        }
    );
});

/**
 * Check if WooCommerce is active
 */
function sl_payments_check_woocommerce() {
    if (!class_exists('WooCommerce')) {
        add_action('admin_notices', function() {
            ?>
            <div class="notice notice-error">
                <p><?php _e('DBG Payments requires WooCommerce to be installed and activated.', 'secondlife-payments'); ?></p>
            </div>
            <?php
        });
    }
}
add_action('admin_init', 'sl_payments_check_woocommerce');

/**
 * Handle payment data from WooCommerce Blocks Store API
 * This ensures the SL name data is available during checkout processing
 */
add_action('woocommerce_store_api_checkout_update_order_from_request', function($order, $request) {
    $payment_method = $request['payment_method'] ?? '';
    
    if ($payment_method !== 'secondlife') {
        return;
    }
    
    $payment_data = $request['payment_data'] ?? array();
    
    // Convert payment_data array to key-value pairs
    $data = array();
    foreach ($payment_data as $item) {
        if (isset($item['key']) && isset($item['value'])) {
            $data[$item['key']] = $item['value'];
        }
    }
    
    // Get SL name from payment data
    $first_name = sanitize_text_field($data['sl_first_name'] ?? '');
    $last_name = sanitize_text_field($data['sl_last_name'] ?? '');
    
    if ($first_name && $last_name) {
        $sl_name = $first_name . ' ' . $last_name;
        $order->update_meta_data('_sl_buyer_name', $sl_name);
        $order->update_meta_data('_sl_first_name', $first_name);
        $order->update_meta_data('_sl_last_name', $last_name);
        
        // Also set in POST for gateway processing
        $_POST['sl_first_name'] = $first_name;
        $_POST['sl_last_name'] = $last_name;
    }
}, 10, 2);
